/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.util;

import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;

/**
 * A weak hash set that is backed by a weak hash map. All methods are
 * forwarded to the backing map's keySet, except addAll which uses the
 * super implementation from <code>AbstractSet</code>.
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 * 
 * @param <E> The type contained in this set.
 * @see WeakHashMap 
 */
public final class WeakHashSet<E> extends AbstractSet<E> {
    
    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////
    /////////                                                    /////////
    /////////   NOTE: In java 1.6, the Collections class         /////////
    /////////   contains a utility method for obtaining a set    /////////
    /////////   from a hash map (newSetFromMap(...). When g9     /////////
    /////////   upgrades to Java 6, the implementation in this   /////////
    /////////   class should be skipped, and the getInstance     /////////
    /////////   method should forward to that class instead      /////////
    /////////   (using a weak hash map).                         /////////    
    /////////                                                    /////////    
    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////
    
    /** The backing map */
    private final Map<E, Boolean> map;
    
    /**
     * Get an instance of an empty, weak, hash set.
     * @param <E> the type of the elements contained in the set.
     * @return a weak hash set.
     */
    public static final <E> Set<E> getInstance() {
        return new WeakHashSet<E>(); 
    }
    
    /**
     * Construct a new weak hash set.
     */
    private WeakHashSet() {
        map = new WeakHashMap<E, Boolean>();
    }

    @Override
    public boolean add(E o) { 
        return map.put(o, Boolean.TRUE) == null; 
    }

    @Override
    public void clear() { 
        map.clear(); 
    }

    @Override
    public boolean contains(Object o) {
        return map.containsKey(o);
    }

    @Override
    public boolean containsAll(Collection<?> c) {
        return map.keySet().containsAll(c);
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public Iterator<E> iterator() {
        return map.keySet().iterator();
    }

    @Override
    public boolean remove(Object o) {
        return map.remove(o) != null;
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        return map.keySet().retainAll(c);
    }

    @Override
    public int size() {
        return map.size();
    }

    @Override
    public Object[] toArray() {
        return map.keySet().toArray();
    }

    @Override
    public <T> T[] toArray(T[] a) {
        return map.keySet().toArray(a);
    }

    @Override
    public String toString() {
        return map.keySet().toString();
    }
    
    @Override
    public int hashCode()             { 
        return map.keySet().hashCode(); 
    }
    
    @Override
    public boolean equals(Object o)   { 
        return map.keySet().equals(o); 
    }
}
