/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

/**
 * Numeric substitution variable in a message text.
 */
class NumVarPart implements ITextPart {

    /** Pattern for a linefeed. */
    private final static String LF_REGEXP = "\\\\n";

    /** The variable number for this variable. */
    private final int varNo;

    /** Default value if variable empty. Never null. */
    private final String defVal;

    /** Prefix if variable is not empty. Never null. */
    private final String prefix;

    /** Suffix if variable is not empty. Never null. */
    private final String suffix;

    /**
     * Default constructor for NumVarPart.
     * 
     * @param pVarNo Variable number. Must be &gt;0
     * @param pPfx Prefix if variable given
     * @param pSfx Suffix if variable given
     * @param pDfl Default if variable not given.
     * @throws IllegalArgumentException If pVarNo&lt;0
     */
    public NumVarPart(final int pVarNo, final String pPfx, final String pSfx,
                      final String pDfl) {
        super();
        if (pVarNo <= 0) {
            throw new IllegalArgumentException("NumVarPart.varNo");
        }
        this.varNo = pVarNo;
        if (pDfl == null) {
            this.defVal = "";
        } else {
            this.defVal = pDfl;
        }
        if (pPfx == null) {
            this.prefix = "";
        } else {
            this.prefix = pPfx;
        }
        if (pSfx == null) {
            this.suffix = "";
        } else {
            this.suffix = pSfx;
        }
    }

    /**
     * Appends this variable to the message. If args[varNo-1] exists, prefix +
     * args[varNo-1] + suffix are added. Otherwise default is added.
     * 
     * @param sb Text to append to
     * @param singleLine Ignored
     * @param args Current variable set
     */
    @Override
    public void appendPart(final StringBuffer sb, final boolean singleLine,
            final Object[] args) {
        String contrib;
        final String lfRepl= singleLine? " " 
                                       : System.getProperty("line.separator");
        final int varIndex= this.varNo-1;  // <1> corresponds to args[0] etc
        if (args == null || varIndex >= args.length
                         || args[varIndex] == null) {
            sb.append(this.defVal.replaceAll(LF_REGEXP, lfRepl));
        } 
        else {
            contrib = "" + args[varIndex];
            if ("".equals(contrib)) {
                sb.append(this.defVal.replaceAll(LF_REGEXP, lfRepl));
            } 
            else {
                sb.append(this.prefix.replaceAll(LF_REGEXP, lfRepl));
                sb.append(contrib);
                sb.append(this.suffix.replaceAll(LF_REGEXP, lfRepl));
            }
        }
    }

    @Override
    public String toString() {
        final StringBuffer sb = new StringBuffer("NumVar ");
        sb.append(this.varNo);
        if (!"".equals(this.prefix)) {
            sb.append(" pfx=");
            sb.append(this.prefix);
        }
        if (!"".equals(this.suffix)) {
            sb.append(" sfx=");
            sb.append(this.suffix);
        }
        if (!"".equals(this.defVal)) {
            sb.append(" def=");
            sb.append(this.defVal);
        }
        return sb.toString();
    }

}
