/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.os;

import java.util.List;
import java.util.Map;

import no.g9.support.Visitable;
import no.g9.support.Visitor;

/**
 * Representation of a node in the object selection.
 * 
 * @param <T> the domain class of the object selection node.
 */
public interface OSRole<T> extends Visitable<OSRole<?>> {

    /**
     * Adds a child to this (parent) node.
     * 
     * @param child the child to add
     * @throws IllegalArgumentException if this is not the parent of the child
     *             or the child is already added.
     */
    void addChild(OSRole<?> child);

    /**
     * If possible, returns a down-casted reference to the supplied object. The
     * reference is of type T.
     * 
     * @param obj the object to downcast
     * @return a casted version of the supplied object, or <code>null</code> if
     *         the object is not assignable to type T.
     */
    T castToType(Object obj);

    /**
     * Creates a new domain instance, using the default (empty) constructor.
     * 
     * @return the domain instance.
     */
    T createNewInstance();

    /**
     * Returns an array of the attribute constants for this object selection
     * node
     * 
     * @return the attribute constants
     */
    AttributeConstant[] getAttributeConstants();

    /**
     * Gets a map of attribute values from the domain object.
     * 
     * @param domainInstance the domain instance
     * @return the map of attribute values.
     */
    Map<AttributeConstant, Object> getAttributeValues(Object domainInstance);

    /**
     * Gets an unmodifiable list of child nodes.
     * 
     * @return the list of child nodes.
     */
    List<OSRole<?>> getChildren();

    /**
     * Gets the child role for the given child role constant.
     * 
     * @param childRole the child role constant
     * @return the child role, or null if not found
     */
    OSRole<?> getChild(RoleConstant childRole);

    /**
     * Gets the domain class object.
     * 
     * @return the domain class object.
     */
    Class<T> getDomainClass();

    /**
     * Gets the main key of the object selection role
     * 
     * @return the main key
     */
    Key getMainKey();

    /**
     * Gets the parent of this node.
     * 
     * @return the parent
     */
    OSRole<?> getParent();

    /**
     * Gets the root of this role (roots return self).
     * @return the root role of this role.
     */
    OSRole<?> getRoot();

    /**
     * Returns the keys for this role. The first key in the list is the primary
     * key.
     * 
     * @return a list of keys.
     */
    List<Key> getKeys();

    /**
     * Gets the role name of this node.
     * 
     * @return the roleName
     */
    RoleConstant getRoleConstant();

    /**
     * Returns a map of attributes values. The map key is the attribute name and
     * the value is the attribute value.
     * 
     * @param instance the domain instance
     * @param attributes a string array of attribute names
     * @return a map of attribute names and values
     */
    Map<AttributeConstant, Object> getValues(Object instance,
            AttributeConstant[] attributes);

    /**
     * Gets the specified attribute value from the domain object.
     * 
     * @param domainObject the domain object
     * @param attribute the constant denoting the attribute.
     * @return the value
     */
    Object getValue(Object domainObject, AttributeConstant attribute);

    /**
     * Gets the specified associated relation from the domain object.
     * 
     * @param domainObject the domain object that holds the relation.
     * @param role the constant enum denoting the role.
     * @return the association
     */
    Object getRelation(Object domainObject, RoleConstant role);

    /**
     * Sets the specified attribute value on the domain object.
     * 
     * @param domainObject the domain object
     * @param attribute the constant denoting the attribute.
     * @param value the value to set
     */
    void setValue(Object domainObject, AttributeConstant attribute, Object value);

    /**
     * Sets the specified relation on the domain object
     * 
     * @param domainInstance the domain object
     * @param relation the related object
     * @param role the constant denoting the relation
     */
    void setRelation(Object domainInstance, Object relation, RoleConstant role);
    
    /**
     * Updates the specified relation. If the specified role is one-related, this
     * is the same as {@link #setRelation(Object, Object, RoleConstant)}. If the
     * specified role is many-related, the relation collection is updated. 
     * 
     * @param domainInstance The relation owner.
     * @param relation the relation member.
     * @param role the role of the member relation.
     */
    void updateRelation(Object domainInstance, Object relation, RoleConstant role);

    /**
     * Get the relation type enumeration for this role.
     * 
     * @return the relation type.
     */
    RelationType getRelationType();

    /**
     * Get the role's cardinality.
     * 
     * @return the cardinality enumeration of this role.
     */
    RelationCardinality getRelationCardinality();

    /**
     * Returns the role name of this os node.
     */
    @Override
    String toString();

    /**
     * Returns <code>true</code> if the relation from this role to the parent is
     * navigable.
     * 
     * @return <code>true</code> if the relation to parent is navigable.
     */
    boolean isNavigableToParent();

    /**
     * Sets the main key.
     * 
     * @param mainKey the mainKey to set
     */
    void setMainKey(Key mainKey);

    /**
     * Compares two domain instances using the specified key.
     * <p>
     * Note that both domain instances must be assignable to the type of the
     * object selection role, otherwise a class cast exception is thrown.
     * 
     * @param keyToUse the key to use when comparing the two instances.
     * @param anObject the first object to compare
     * @param anOtherObject the other object to compare
     * @return <code>true</code> if all attributes in the keys are equal in both
     *         domain objects.
     * @throws ClassCastException if either domain object can not be assigned to
     *             the object selection role type.
     */
    boolean equalsUsingKey(Key keyToUse, Object anObject, Object anOtherObject);

    /**
     * Test if this role is ancestor of <code>possibleHeir</code>. A role
     * <em>A</em> is an ancestor of another role <em>B</em> if the <em>A</em>
     * role can be reached by navigating up (in direction of the root) from the
     * <em>B</em> role.
     * 
     * @param possibleHeir the role that is possible a child or grand (grand..)
     *            child of this role.
     * @return true if this role is an ancestor of <code>possibleHeir</code>.
     */
    boolean isAncestorOf(RoleConstant possibleHeir);

    /**
     * Test if an attribute is a related attribute of this object selection
     * role. An attribute is considered <em>related</em> if the attribute path
     * contains "this" role.
     * 
     * @param attribute the attribute to check the relation to.
     * @return <code>true</code> if the attribute path contains "this" role.
     */
    boolean isRelated(AttributeConstant attribute);

    /**
     * Test if this role is an up-related child.
     * 
     * @return <code>true</code> if this role is up-related.
     */
    boolean isUpRelated();

    /**
     * Test if the association to the parent role is a many-association.
     * 
     * @return <code>true</code> if the association to parent is many-related.
     */
    boolean isParentMany();
    
    /**
     * Test if the role is persistent.
     * 
     * @return <code>true</code> if the role is persistent.
     */
    boolean isPersistent();
    
    /**
     * Get the attribute constant that matches the specified attribute name.
     * @param attributeName the attribute name
     * @return the attribute constant representing the specified attribute.
     */
    AttributeConstant getAttributeConstant(String attributeName);

    /**
     * Test if the association to this role from the parent role is a
     * many-association.
     * 
     * @return <code>true</code> if the association to this role from the parent
     *         is a many-relation.
     */
    boolean isMany();

    /**
     * Test if this is a root node.
     * 
     * @return <code>true</code> if this is a root node in the object selection
     *         tree.
     */
    boolean isRoot();
    
    /**
     * Visit this node and all child nodes (recursively).
     * @param visitor the visitor to visit the object selection roles.
     */
    void visitBranch(Visitor<OSRole<?>> visitor);
    
}
