/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service;

import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import no.esito.log.Logger;
import no.esito.spring.aspect.ExceptionHandlerAspect;
import no.g9.service.print.ExportService;
import no.g9.service.print.G9Print;
import no.g9.service.print.PrintService;
import no.g9.support.Pathfinder;
import no.g9.support.xml.XmlConverter;

import org.springframework.aop.framework.ProxyFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.core.SpringVersion;

/**
 * The class is the common client-side proxy super class for Spring usage (as
 * remote service).
 * <p>
 * Bean definitions are read from three types of xml-files.
 * <ol>
 * <li>Always present - these are the xml files that are part of the framework.
 * <li>Optional with known name - these are xml files that have a pre-defined
 * name, but might not be present, e.g. <code>G9Service.xml</code>
 * <li>Additional files - these are xml files that have been supplied runtime.
 * </ol>
 */
public class G9Spring {

    /** The default bean definitions files. */
    private static final String[] DEFAULT_BEAN_DEFINITIONS = {
            "config/ServicesConfiguration.xml", // common
            "config/MessagesConfiguration.xml", // common
            "config/JGrapeServicesConfiguration.xml", // jgrape
            "config/SwingServicesConfiguration.xml", // swing
            "config/ICEfacesServicesConfiguration.xml", // ICEfaces
            "config/jVineServicesConfiguration.xml" }; // jVine

    /** The optional bean definitions files. */
    private static final String[] OPTIONAL_BEAN_DEFINITIONS =
            { "G9Dataaccess.xml", "ApplicationBeans.xml", "JouteurApplicationBeans.xml", "G9Service.xml" };

    private static List<String> additionalBeanDefinitions =
            new ArrayList<String>();

    private static Logger logger = Logger.getLogger(G9Spring.class);

    private static ApplicationContextReference applicationContextReference=new ApplicationContextReference();

    /**
     * Reference class used to allow for changing the ApplicationContext by installing a reimplementation of this class.
     */
    public static class ApplicationContextReference{
    	private ApplicationContext context;

		/**
		 * @return the applicationContext
		 */
		public ApplicationContext getApplicationContext() {
			return context;
		}

		/**
		 * @param applicationContext the applicationContext to set
		 */
		public void setApplicationContext(ApplicationContext applicationContext) {
			this.context = applicationContext;
		}

		/**
		 * Replace the class with this reference.
		 */
		final protected void install(){
			if(applicationContextReference!=null)
				this.setApplicationContext(applicationContextReference.getApplicationContext());
			applicationContextReference=this;
		}
    }

    private static Set<Class<?>> shouldUseAspect;

    static {
        Class<?>[] tmp =
                new Class[] { JGrapeService.class,
                        XmlConverter.class, PrintService.class,
                        ExportService.class, G9Print.class };
        shouldUseAspect = new HashSet<Class<?>>();
        shouldUseAspect.addAll(Arrays.asList(tmp));
    }

    /**
     * Prohibit instantiation
     */
    protected G9Spring() {
        // only subclasses should instantiate this class.
    }
    
    /**
     * @return array of default bean definitions.
     */
    public static String[] getDefaultBeanDefinitions() {
        return DEFAULT_BEAN_DEFINITIONS.clone();
    }
    
    /**
     * @return array of optional bean definitions.
     */
    public static String[] getOptionalBeanDefinitions() {
        return OPTIONAL_BEAN_DEFINITIONS.clone();

    }

    /**
     * Add an additional file name as a spring bean definitions xml file. The
     * file must be on the class path in order for spring to find it. The file
     * must be added <em>before</em> the spring application context is loaded
     * (which happens the first time a bean is accessed), in other words -- do
     * this at the earliest possible time.
     *
     * @deprecated include additional xml files in the G9Service.xml file
     *
     * @param fileName the name of the xml file.
     */
    @Deprecated
    public static void addAdditionalBeanDefinition(String fileName) {
        logger.debug("Adding additional bean definition " + fileName);

        boolean issueWarning = true;
        checkResource(fileName, issueWarning);
        additionalBeanDefinitions.add(fileName);
        logger.info("Added spring bean definition file " + fileName);
    }

    /**
     * Gets the specified spring bean.
     *
     * @param <T> the interface type of the bean
     * @param serviceInterface the interface class that the bean implements
     * @param beanID the bean id used by Spring to identify the bean
     * @return the Spring bean
     */
    public static <T> T getBean(Class<T> serviceInterface, String beanID) {
        Object bean = getApplicationContext().getBean(beanID);
        if (logger.isDebugEnabled()) {
            logger.debug("Got bean " + beanID);
        }
        if (logger.isTraceEnabled()) {
            logger.trace(beanID + " is of type " + bean.getClass().getName());
        }

        if (shouldUseAspect.contains(serviceInterface)) {
            ProxyFactory factory = new ProxyFactory(bean);
            factory.addAdvice(new ExceptionHandlerAspect());
            bean = factory.getProxy();
        }

        return serviceInterface.cast(bean);
    }

    /**
     * Gets the specified spring bean.
     *
     * @param <T> the type of the bean.
     * @param beanID the bean id used by Spring to identify the bean.
     * @return the Spring bean.
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBean(String beanID) {
        Object bean = getApplicationContext().getBean(beanID);
        if (logger.isDebugEnabled()) {
            logger.debug("Got bean " + beanID);
        }
        if (logger.isTraceEnabled()) {
            logger.trace(beanID + " is of type " + bean.getClass().getName());
        }
        return (T) bean;
    }

    /**
     * Get the spring application context.
     *
     * @return the applicationContext set up with the bean definitions.
     */
    public static synchronized ApplicationContext getApplicationContext() {
        if (G9Spring.applicationContextReference.getApplicationContext() == null) {
            loadApplicationContext();
        }
        // The context might have changed
        return G9Spring.applicationContextReference.getApplicationContext();
    }

    /**
     * Load the spring application context if needed. This is only performed the
     * first time, viz. when the application context has not been previously
     * loaded.
     */
    public static synchronized void loadApplicationContext() {
        if (G9Spring.applicationContextReference.getApplicationContext() == null) {

            String[] beanDefinitions = getBeanDefinitions();
            if (logger.isInfoEnabled()) {
                logger.info("Spring version used: "
                        + SpringVersion.getVersion());

                String msg = "Instantiating spring application context using [";
                for (int i = 0; i < beanDefinitions.length - 1; i++) {
                    msg += beanDefinitions[i] + ", ";
                }
                msg += beanDefinitions[beanDefinitions.length - 1] + "]";
                logger.info(msg);

            }
            ClassPathXmlApplicationContext applicationContext = new ClassPathXmlApplicationContext(beanDefinitions, false);
            G9Spring.applicationContextReference.setApplicationContext(applicationContext);
            applicationContext.refresh();
        }
    }

    private static boolean checkResource(String fileName, boolean issueWarning) {
        URL url = Pathfinder.getResourceReference(null, fileName);
        logger.debug("Resource url: " + url);
        if (url != null) {
            logger.info("Found resource " + fileName + " using uri: " + url);
        } else if (issueWarning) {
            logger.warn("Could not locate resource " + fileName);
        }
        return url != null;
    }

    /**
     * @return an array of Spring resource files to be used for the application context
     */
    public static String[] getBeanDefinitions() {
        List<String> presentResouces = new ArrayList<String>();

        // Add default bean definitions, no warning if missing...
        for (String fileName : DEFAULT_BEAN_DEFINITIONS) {
            boolean issueWarning = false;
            if (checkResource(fileName, issueWarning)) {
                presentResouces.add(fileName);
            }
        }

        // Add optional bean definitions, issue warning if missing...
        for (String fileName : OPTIONAL_BEAN_DEFINITIONS) {
            boolean issueWarning = true;
            if (checkResource(fileName, issueWarning)) {
                presentResouces.add(fileName);
            }
        }

        // Add additional bean definitions, warning already issued if missing.
        presentResouces.addAll(additionalBeanDefinitions);

        String[] resources = presentResouces.toArray(new String[0]);
        return resources;
    }

}
