/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service.print;

import java.io.Serializable;
import java.net.URI;
import java.util.Map;
import org.w3c.dom.Document;

/**
 * The PrintContext is a carrier of data and parameters, used by implementations
 * of the PrintService.
 */
public class PrintContext implements Serializable {

    private String dialogName;
    private Document document;
    private URI file;
    private String xml;
    private String printerName;
    private Map<String, Object> params;

    /**
     * Construct a new Print context
     * */
    public PrintContext() {
        // Empty
    }

    /**
     * Create a PrintContext based on a Document.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param document Data to be printed formed in a Document
     */
    public PrintContext(final String dialogName, final Document document) {
        this(dialogName, document, null, null, null, null);
    }

    /**
     * Create a PrintContext based on a Document. The given printer name will be
     * used for printing.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param document Data to be printed formed in a Document
     * @param printerName The name of the printer to be used
     */
    public PrintContext(final String dialogName, final Document document,
            final String printerName) {
        this(dialogName, document, null, null, printerName, null);
    }

    /**
     * Create a PrintContext based on a file given as an URI.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param file Data to be printed formed in a file (XML file)
     */
    public PrintContext(final String dialogName, final URI file) {
        this(dialogName, null, file, null, null, null);
    }

    /**
     * Create a PrintContext based on a file given as an URI.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param file Data to be printed formed in a file (XML file)
     * @param printerName The name of the printer to be used
     */
    public PrintContext(final String dialogName, final URI file,
            final String printerName) {
        this(dialogName, null, file, null, printerName, null);
    }

    /**
     * Create a PrintContext based on a Document. The given printer name will be
     * used for printing.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param document Data to be printed formed in a Document
     * @param params Additional print parameters
     */
    public PrintContext(final String dialogName, final Document document,
            final Map<String, Object> params) {
        this(dialogName, document, null, null, null, params);
    }

    /**
     * Create a PrintContext based on a Document. The given printer name will be
     * used for printing.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param document Data to be printed formed in a Document
     * @param printerName The name of the printer to be used
     * @param params Additional print parameters
     */
    public PrintContext(final String dialogName, final Document document,
            final String printerName, final Map<String, Object> params) {
        this(dialogName, document, null, null, printerName, params);
    }

    /**
     * Create a PrintContext based on a file given as an URI.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param file Data to be printed formed in a file (XML file)
     * @param params Additional print parameters
     */
    public PrintContext(final String dialogName, final URI file,
            final Map<String, Object> params) {
        this(dialogName, null, file, null, null, params);
    }

    /**
     * Create a PrintContext based on a file given as an URI.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param file Data to be printed formed in a file (XML file)
     * @param printerName The name of the printer to be used
     * @param params Additional print parameters
     */
    public PrintContext(final String dialogName, final URI file,
            final String printerName, final Map<String, Object> params) {
        this(dialogName, null, file, null, printerName, params);
    }

    /**
     * Create a PrintContext based on an XML string.
     *
     * @param dialogName The name of the print dialog (print template)
     * @param xml Data to be printed formed in an XML string
     * @param printerName The name of the printer to be used
     */
    public PrintContext(final String dialogName, final String xml,
            final String printerName) {
        this(dialogName, null, null, xml, printerName, null);
    }

    private PrintContext(final String dialogName, final Document document,
            final URI file, final String xml, final String printerName,
            final Map<String, Object> params) {
        this.dialogName = dialogName;
        this.document = document;
        this.file = file;
        this.xml = xml;
        this.printerName = printerName;
        this.params = params;
    }

    /**
     * Check whether this PrintContext is valid for printing.
     *
     * @return true if valid
     */
    public boolean isValid() {
        return dialogName != null && dialogName.length() > 0
                && (document != null || file != null || xml != null);
    }

    @Override
    public String toString() {
        return "[PrintContext | " + dialogName + "]";
    }

    /**
     * @return (missing javadoc)
     */
    public String getDialogName() {
        return dialogName;
    }

    /**
     * @param dialogName (missing javadoc)
     */
    public void setDialogName(final String dialogName) {
        this.dialogName = dialogName;
    }

    /**
     * @return (missing javadoc)
     */
    public URI getFile() {
        return file;
    }

    /**
     * @param file (missing javadoc)
     */
    public void setFile(final URI file) {
        this.file = file;
    }

    /**
     * @return (missing javadoc)
     */
    public Document getDocument() {
        return document;
    }

    /**
     * @param document (missing javadoc)
     */
    public void setDocument(final Document document) {
        this.document = document;
    }

    /**
     * @return (missing javadoc)
     */
    public String getXml() {
        return xml;
    }

    /**
     * @param xml (missing javadoc)
     */
    public void setXml(final String xml) {
        this.xml = xml;
    }

    /**
     * @return (missing javadoc)
     */
    public String getPrinterName() {
        return printerName;
    }

    /**
     * @param printerName (missing javadoc)
     */
    public void setPrinterName(final String printerName) {
        this.printerName = printerName;
    }

    /**
     * @return (missing javadoc)
     */
    public Map<String, Object> getParams() {
        return params;
    }

    /**
     * @param params (missing javadoc)
     */
    public void setParams(final Map<String, Object> params) {
        this.params = params;
    }

}
