/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * Utility class facilitating the creation of new objects using reflection.
 * The ObjectFactory only contains static methods and is not intended for
 * instantiation.
 */
public class ObjectFactory {

    /**
     * Private constructor.
     */
    private ObjectFactory() {
        // Empty
    }

    /**
     * Creates a new instance of the object with the specified class name, using
     * the default constructor.
     *
     * @param className the fully qualified name of the class.
     * @return the created instance.
     */
    public static Object newObject(String className) {
        Object object = null;
        String msgID = null;
        Exception ex = null;
        try {
            Class<?> classDefinition = Class.forName(className);
            object = classDefinition.newInstance();
        } catch (InstantiationException e) {
            msgID = CRuntimeMsg.R_CANT_CREATE_CLASS;
            ex = e;
        } catch (IllegalAccessException e) {
            msgID = CRuntimeMsg.R_CANT_CREATE_CLASS;
            ex = e;
        } catch (ClassNotFoundException e) {
            msgID = CRuntimeMsg.R_CLASS_NOT_FOUND;
            ex = e;
        } finally {
            if (msgID != null) {
                Object[] msgArgs = { ObjectFactory.class, className, ex };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
        return object;
    }

    /**
     * Creates a new instance of the object with the specified class name, using
     * the suplied arguments to the constructor.
     * @param className the name of the class
     * @param params the (formal) argument list of the class constructor
     * @param args the (actual) argument list of the class constructor.
     * @return the created instance
     */
    public static Object newObject(String className, Class<?>[] params, Object[] args) {
        String msgID = null;
        Exception ex = null;
        try {
            Class<?> classDefinition = Class.forName(className);
            Constructor<?> classConstructor = classDefinition.getConstructor(params);
            // Work-around for bug #4533479 in java
            classConstructor.setAccessible(true);
            return classConstructor.newInstance(args);
        } catch (ClassNotFoundException e) {
            msgID = CRuntimeMsg.R_CLASS_NOT_FOUND;
            ex = e;
        } catch (NoSuchMethodException e) {
            msgID = CRuntimeMsg.R_CANT_CREATE_CLASS;
            ex = e;
        } catch (InstantiationException e) {
            msgID = CRuntimeMsg.R_CANT_CREATE_CLASS;
            ex = e;
        } catch (IllegalAccessException e) {
            msgID = CRuntimeMsg.R_CANT_CREATE_CLASS;
            ex = e;
        } catch (InvocationTargetException e) {
            msgID = CRuntimeMsg.R_CANT_CREATE_CLASS;
            ex = e;
        }
        // If here, we had an exception...
        Object[] msgArgs = { ObjectFactory.class, className, ex };
        Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9ClientFrameworkException(ex, msg);
    }
}
