/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.convert;

import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 * Exception for converters
 *
 */
public class ConvertException extends Exception {

    private Message errMsg;
    private ConvertContext convertContext;
    private Object provokingValue;


    /**
     * @param convertContext The conversion context
     * @param provokingValue The value related to the vonversion attempt
     * @param cause Cause of this exception
     */
    public ConvertException(ConvertContext convertContext, Object provokingValue, Throwable cause) {
        super(cause);
        this.convertContext= convertContext;
        this.provokingValue= provokingValue;
        this.setErrMsg(MessageSystem.getMessageFactory().getMessage(
                CRuntimeMsg.UNHANDLED_EXCEPTION, cause.getLocalizedMessage()));
        this.getErrMsg().setException(cause);
    }

    /**
     * @param convertContext The conversion context
     * @param provokingValue The value related to the vonversion attempt
     * @param message Exception message as String
     * @param cause Cause of this exception
     */
    public ConvertException(ConvertContext convertContext, Object provokingValue,
            String message, Throwable cause) {
        super(message, cause);
        this.convertContext = convertContext;
        this.provokingValue = provokingValue;
        this.setErrMsg(MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.UNHANDLED_EXCEPTION, message));
        this.getErrMsg().setException(cause);
    }

    /**
     * @param convertContext The conversion context
     * @param provokingValue The value related to the vonversion attempt
     * @param message Exception message as String
     */
    public ConvertException(ConvertContext convertContext, Object provokingValue, String message) {
        super(message);
        this.convertContext = convertContext;
        this.provokingValue = provokingValue;
        this.setErrMsg(MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.UNHANDLED_EXCEPTION, message));
    }

    /**
     * @param convertContext The conversion context
     * @param provokingValue The value related to the vonversion attempt
     * @param errMsg Exception message
     */
    public ConvertException(ConvertContext convertContext, Object provokingValue, final Message errMsg) {
        super(errMsg != null ? errMsg.getMessageText() : null);
        this.convertContext = convertContext;
        this.provokingValue = provokingValue;
        this.setErrMsg(errMsg);
    }

    /**
     * @param convertContext The conversion context
     * @param provokingValue The value related to the vonversion attempt
     * @param cause Cause of this exception
     * @param errMsg Exception message
     */
    public ConvertException(ConvertContext convertContext,
            Object provokingValue, Throwable cause, final Message errMsg) {
        super(errMsg != null ? errMsg.getMessageText() : null, cause);
        this.convertContext = convertContext;
        this.provokingValue = provokingValue;
        this.setErrMsg(errMsg);
        this.getErrMsg().setException(cause);
    }

    /**
     * Store a Message.
     *
     * @param errMsg New value of errMsg.
     * @see #getErrMsg()
     */
    public final void setErrMsg(final Message errMsg) {
        this.errMsg = errMsg;
    }

    /**
     * Returns the current error message. NOTE: original object, not copied .
     *
     * @return The stored Message object, if any.
     * @see #setErrMsg(Message)
     */
    public final Message getErrMsg() {
        return this.errMsg;
    }

    /**
     * Returns the conversion context.
     * @return the conversion context
     */
    public ConvertContext getConvertContext() {
        return convertContext;
    }

    /**
     * @return the provoking Value
     */
    public Object getProvokingValue() {
        return provokingValue;
    }

}
