/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;

import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.support.G9Enumerator;

import org.springframework.beans.BeanInstantiationException;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.InvalidPropertyException;
import org.springframework.beans.PropertyAccessException;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.converter.ConverterFactory;
import org.springframework.util.ReflectionUtils;

/**
 * A factory for conversion of G9Enum objects to Java enums.
 *
 * <strong>WARNING:</strong> Although this class is public, it should not be treated as part of the public API, as it
 * might change in incompatible ways between releases (even patches).
 *
 * <p>
 * <strong>NOTE:</strong> This factory depends on a static method called <code>ordinalToEnum</code> on the target type.
 */
@SuppressWarnings("rawtypes")
public class JavaEnumToG9EnumConverterFactory implements ConverterFactory<Enum, G9Enumerator> {

    /** The property name to get the ordinalValue from the source Enum. */
    private static final String GET_ORDINAL_VALUE_PROPERTY_NAME = "value";

    /*
     * (non-Javadoc)
     *
     * @see org.springframework.core.convert.converter.ConverterFactory#getConverter (java.lang.Class)
     */
    @Override
    public <T extends G9Enumerator> Converter<Enum, T> getConverter(Class<T> targetType) {
        return new JavaEnumToG9EnumConverter<T>(targetType);
    }

    /**
     * Converter class for conversion from a Java enum to a G9Enumerator.
     *
     * @param <T>
     *            the generic type
     */
    private final static class JavaEnumToG9EnumConverter<T extends G9Enumerator> implements Converter<Enum, T> {

        /** The enum type. */
        private Class<T> g9Type;

        /** The static factory method for G9Enumerator. */
        Constructor<T> g9Constructor = null;

        /**
         * Instantiates a new java enum to G9Enumerator converter.
         *
         * @param g9Type
         *            the G9Enumerator type
         */
        public JavaEnumToG9EnumConverter(Class<T> g9Type) {
            this.g9Type = g9Type;
            try {
                this.g9Constructor = g9Type.getConstructor(int.class);
                ReflectionUtils.makeAccessible(this.g9Constructor);
            } catch (SecurityException se) {
                handleException(g9Type.getSimpleName() + "(int.class)", se);
            } catch (NoSuchMethodException nm) {
                handleException(g9Type.getSimpleName() + "(int.class)", nm);
            }
        }

        /*
         * (non-Javadoc)
         *
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        public T convert(Enum source) {
            try {
                if(source == null) {
                    return null;
                }
                Field field =  ReflectionUtils.findField(source.getClass(), GET_ORDINAL_VALUE_PROPERTY_NAME);
                ReflectionUtils.makeAccessible(field);
                Integer ordinalValue = (Integer) ReflectionUtils.getField(field, source);
                return BeanUtils.instantiateClass(this.g9Constructor, ordinalValue);
            } catch (InvalidPropertyException i) {
                handleException(GET_ORDINAL_VALUE_PROPERTY_NAME, i);
            } catch (PropertyAccessException i) {
                handleException(GET_ORDINAL_VALUE_PROPERTY_NAME, i);
            } catch (BeanInstantiationException bi) {
                handleException(this.g9Type.getSimpleName() + "(int.class)", bi);
            } catch (BeansException i) {
                handleException(GET_ORDINAL_VALUE_PROPERTY_NAME, i);
            }
            return null;
        }

        private void handleException(String methodOrField, Exception ex) {
            Object[] msgArgs = { g9Type.getClass(), methodOrField, this.getClass(), ex.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                            msgArgs);
            throw new G9ServiceException(ex, msg);
        }
    }
}
