/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import no.g9.exception.G9BaseException;
import no.g9.support.Numeric;

import org.springframework.core.convert.converter.Converter;

/**
 * This class contains {@link Converter} implementations used in data type mappings for the transport objects.
 */
public class TransportTypeConversion {

    /**
     * Converter from {@link Numeric} to {@link BigDecimal}.
     */
    public final static class NumericToBigDecimal implements Converter<Numeric, BigDecimal> {

        /*
         * (non-Javadoc)
         * 
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        public BigDecimal convert(Numeric source) {
            return source.getValue();
        }

    }

    /**
     * Converter from {@link BigDecimal} to {@link Numeric}.
     */
    public final static class BigDecimalToNumeric implements Converter<BigDecimal, Numeric> {

        /*
         * (non-Javadoc)
         * 
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        public Numeric convert(BigDecimal source) {
            return new Numeric(source, source.scale());
        }

    }

    /**
     * Converter from {@link Date} to {@link XMLGregorianCalendar}.
     */
    public final static class DateToXMLGregorianCalendar implements Converter<Date, XMLGregorianCalendar> {

        /*
         * (non-Javadoc)
         * 
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        public XMLGregorianCalendar convert(Date source) {
            Calendar cal = Calendar.getInstance();
            cal.setTime(source);
            try {
                return DatatypeFactory.newInstance().newXMLGregorianCalendar((GregorianCalendar) cal);
            } catch (DatatypeConfigurationException e) {
                throw new G9BaseException("Failed to convert date to XMLGregorianCalendar", e);
            }
        }

    }

    /**
     * Converter from {@link XMLGregorianCalendar} to {@link Date}.
     */
    public final static class XMLGregorianCalendarToDate implements Converter<XMLGregorianCalendar, Date> {

        /*
         * (non-Javadoc)
         * 
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        public Date convert(XMLGregorianCalendar source) {
            return source.toGregorianCalendar().getTime();
        }
    }

}
