/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.domain;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Stack;

import no.esito.util.EnumUtil;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.support.G9Enumerator;
import no.g9.support.Numeric;
import no.g9.support.TypeTool;

/**
 * This class acts as a proxy for domain objects, and is a concrete
 * implementation of DomainObjectProxy.
 * @param <T> Type of proxied domain object
 *
 */
public class G9Proxy<T> implements DomainObjectProxy<T> {

    /** Default zero value for byte */
    public final static byte ZERO_VALUE_BYTE = 0;

    /** Default zero value for short */
    public final static short ZERO_VALUE_SHORT = 0;

    /** Default zero value for int */
    public final static int ZERO_VALUE_INT = 0;

    /** Default zero value for char */
    public final static char ZERO_VALUE_CHAR = 0;

    /** Default zero value for long */
    public final static long ZERO_VALUE_LONG = 0;

    /** Default zero value for float */
    public final static float ZERO_VALUE_FLOAT = 0;

    /** Default zero value for double */
    public final static double ZERO_VALUE_DOUBLE = 0;

    /** Default zero value for boolean */
    public final static boolean ZERO_VALUE_BOOLEAN = false;

    /**
     * Map of attributes changed. Key = attributeName (String) Value = Stack of
     * attribute values (wrapped primitives), where the first stack element is
     * the initial value. Lazy initialization.
     */
    private Map<String, Stack<Object>> attributesActedOn;

    /**
     * The domain object being proxied.
     */
    private T domainObject;

    /**
     * The class of the domain object
     */
    private Class<T> domainClass;

    /**
     * Flags wether to use bean-style getters and setters when accessing
     * attributes.
     */
    private boolean useAccessMethods;

    /** Reference to g9 enumerators */
    private G9Enums enums;

    /**
     * Creates a new G9Proxy.
     *
     * @param domainObject the domain object that shall be proxied
     * @param enums the application's enumerators
     * @param useAccessMethods if <code>true</code> variables are accessed using
     *            bean-style getter and setter methods.
     */
    public G9Proxy(T domainObject, G9Enums enums, boolean useAccessMethods) {
        this.domainObject = domainObject;
        domainClass = DomainUtil.getDomainClass(domainObject);
        this.useAccessMethods = useAccessMethods;
        this.enums = enums;
    }

    @Override
    public boolean isClean() {
        return attributesActedOn == null;
    }

    @Override
    public boolean isClean(String attributeName) {
        return isClean() || !attributesActedOn.containsKey(attributeName);
    }

    @Override
    public T getCleanVersionOfObject() {
        if (isClean()) {
            return domainObject;
        }
        G9Proxy<T> cleanProxy = new G9Proxy<T>(createClone(), enums, useAccessMethods);
        Iterator<String> it = attributesActedOn.keySet().iterator();
        while (it.hasNext()) {
            String attributeName = it.next();
            Object value = attributesActedOn.get(attributeName).firstElement();
            cleanProxy.setAttribute(attributeName, value);
        }
        return cleanProxy.getObject();
    }

    @Override
    public T getObject() {
        return domainObject;
    }

    @Override
    public void setAttribute(String attributeName, Object attributeValue) {
        updateAttribute(attributeName, attributeValue);
        if (useAccessMethods) {
            setAttributeUsingMethod(attributeName, attributeValue);
        } else {
            setAttributeDirect(attributeName, attributeValue);
        }
    }

    @Override
    public Object getAttribute(String attributeName) {
        if (isClean() || !attributesActedOn.containsKey(attributeName)) {
            return getCleanVersionOfAttribute(attributeName);
        }
        return attributesActedOn.get(attributeName).peek();
    }

    @Override
    public void reset() {
        while (!isClean()) {
            Iterator<String> it = attributesActedOn.keySet().iterator();

            if (it.hasNext()) {
                String attributeName = it.next();
                resetAttribute(attributeName);
            }
        }
    }

    @Override
    public void resetAttribute(String attributeName) {
        if (isClean(attributeName)) {
            return;
        }
        Object value = attributesActedOn.get(attributeName).firstElement();
        setAttribute(attributeName, value);
        attributesActedOn.remove(attributeName);
        if (attributesActedOn.size() == 0) {
            attributesActedOn = null;
        }
    }

    @Override
    public Object getCleanVersionOfAttribute(String attributeName) {
        if (!isClean(attributeName)) {
            return attributesActedOn.get(attributeName).firstElement();
        }
        return useAccessMethods ? getAttributeUsingMethod(attributeName)
                : getAttributeDirect(attributeName);
    }

    @Override
    public boolean equalsInitialValue(String attributeName, Object value) {
        Object originalValue = getCleanVersionOfAttribute(attributeName);
        if (value instanceof String && originalValue instanceof String) {
            // Here, value != null and originalValue != null
            return ((String) value).trim().equals(
                    ((String) originalValue).trim());
        }
        value = convertStringToObjectType((String) value,
                getAttributeType(attributeName));
        if (value == null) {
            return originalValue == null;
        }
        return value.equals(originalValue);

    }

    /**
     * Updates the specified value and sets the state of this attribute to
     * "clean"
     *
     * @param attributeName the name of the attribute
     * @param value the value to set on the attribute.
     */
    @Override
    public void forceClean(String attributeName, Object value) {
        setAttribute(attributeName, value);
        forceCleanAttribute(attributeName);
    }

    @Override
    public void forceClean(T o) {
        domainObject = o;
        attributesActedOn = null;
    }

    private void forceCleanAttribute(String attributeName) {
        if (attributesActedOn != null) {
            attributesActedOn.remove(attributeName);
            if (attributesActedOn.size() == 0) {
                attributesActedOn = null;
            }
        }
    }

    @Override
    public void copyStatus(DomainObjectProxy<?> fromProxy,
            String fromAttributeName, String toAttributeName) {
        if (fromProxy.isClean(fromAttributeName) && !isClean(toAttributeName)) {
            forceCleanAttribute(toAttributeName);
        } else if (!fromProxy.isClean(fromAttributeName) && isClean(toAttributeName)) {
            updateAttribute(toAttributeName, getAttribute(toAttributeName));
        }
    }

    /**
     * Returns a string representation of this proxy. The string contains
     * information about which class, object and state of the object being
     * proxied.
     *
     * @return a string containing debug infromation
     */
   @Override
    public String toString() {
        return "Proxy for class: " + domainClass.getName() + " -- "
                + ", object: " + domainObject + ", state: "
                + (isClean() ? "clean" : "dirty");
    }

    // ////////////////////////////// PRIVATE METHODS
    // //////////////////////////

    /**
     * Returns the declared type of the specified attribute.
     *
     * @param attributeName the name of the attribute
     * @return class object representing the declared type of the field.
     */
    private Class<?> getAttributeType(String attributeName) {
        if (useAccessMethods) {
            String getMethodName = TypeTool.asBeanGetter(attributeName);
            try {
                Method m = domainClass.getMethod(getMethodName, (Class[]) null);
                return m.getReturnType();
            } catch (NoSuchMethodException e) {
                Object[] msgArgs = { "G9Proxy", getMethodName, domainClass,
                        e.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                        msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(e, msg);
            }
        }

        try {
            Field f = domainClass.getField(attributeName);
            return f.getType();
        } catch (NoSuchFieldException e) {
            Object[] msgArgs = { "G9Proxy", attributeName, domainClass,
                    e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                    msgArgs);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(e, msg);
        }

    }

    /**
     * Gets the value of the specified attribute using direct field access.
     *
     * @param attributeName the name of the attribute
     * @return an object holding the value of the attribute.
     */
    private Object getAttributeDirect(String attributeName) {
        Exception ex = null;
        try {
            Field field = domainClass.getField(attributeName);
            // Work-around for bug #4533479 in java
            field.setAccessible(true);
            return field.get(domainObject);
        } catch (IllegalAccessException e) {
            ex = e;
        } catch (NoSuchFieldException e) {
            ex = e;
        }
        // ex != null
        Object[] msgArgs = {"G9Proxy", attributeName, domainClass, ex.getMessage()};
        Message msg = MessageSystem.getMessageFactory().getMessage(
                CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                msgArgs);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9ClientFrameworkException(ex, msg);
    }

    /**
     * Gets the value of the specified attribute using a bean-style get method.
     *
     * @param attributeName the name of the attribute
     * @return an object holding the value of the attribute.
     */
    private Object getAttributeUsingMethod(String attributeName) {
        Exception ex = null;
        String methodName = TypeTool.asBeanGetter(attributeName);
        try {
            Method getMethod = domainClass.getMethod(methodName, (Class[]) null);
            // Work-around for bug #4533479 in java
            getMethod.setAccessible(true);
            return getMethod.invoke(domainObject, (Object[]) null);
        } catch (IllegalAccessException e) {
            ex = e;
        } catch (InvocationTargetException e) {
            ex = e;
        } catch (NoSuchMethodException e) {
        	ex = e;
        	// Try is
        	methodName = TypeTool.asBeanBooleanGetter(attributeName);
            try {
                Method getMethod = domainClass.getMethod(methodName, (Class[]) null);
                // Work-around for bug #4533479 in java
                getMethod.setAccessible(true);
                return getMethod.invoke(domainObject, (Object[]) null);
            } catch (IllegalAccessException e2) {
                ex = e2;
            } catch (InvocationTargetException e2) {
                ex = e2;
            } catch (NoSuchMethodException e2) {
            	// EMPTY
            }
        }
        // ex != null
        Object[] msgArgs = {"G9Proxy", methodName, domainClass, ex.getMessage()};
        Message msg = MessageSystem.getMessageFactory().getMessage(
                CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                msgArgs);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9ClientFrameworkException(ex, msg);
    }

    /**
     * Updates the specified attribute with the specified value
     *
     * @param attributeName the name of the attribute
     * @param value object holding the new value of the attribute.
     */
    private void updateAttribute(String attributeName, Object value) {
        if (isClean()) {
            attributesActedOn = new HashMap<String, Stack<Object>>();
        }
        Stack<Object> valueStack;
        if (attributesActedOn.containsKey(attributeName)) {
            valueStack = attributesActedOn.get(attributeName);
        } else {
            valueStack = new Stack<Object>();
            valueStack.push(getAttribute(attributeName));
            attributesActedOn.put(attributeName, valueStack);
        }
        valueStack.push(value);
    }

    /**
     * Converts a string to the specified type. In addition to all primitive and
     * wrapper types, Date, G9Enumerator and String is recognized. If the
     * supplied stringValue is either null, empty or consists entirely of
     * blanks, the returned object is given the appropriate zero value. If the
     * type is an object type (e.g. a wrapper class, a Date etc.) the default
     * zero value is <code>null</code>. If the type is a primitive (e.g. int,
     * double etc.) the default zero value is read from the constants defined in
     * this class.
     *
     * @param stringValue a String representation of the value
     * @param type class object representing the desired type.
     * @return an object of the specified type.
     */
    private Object convertStringToObjectType(String stringValue, Class<?> type) {
        Object value = null;
        if (stringValue == null) {
            stringValue = "";
        }
        stringValue = stringValue.trim();
        boolean zeroValue = stringValue.length() == 0;
        if (type == Byte.TYPE) {
            value = zeroValue ? Byte.valueOf(ZERO_VALUE_BYTE) : Byte.valueOf(stringValue);
        } else if (type == Byte.class) {
            value = zeroValue ? null : Byte.valueOf(stringValue);
        } else if (type == Short.TYPE) {
            value = zeroValue ? Short.valueOf(ZERO_VALUE_SHORT) : Short.valueOf(stringValue);
        } else if (type == Short.class) {
            value = zeroValue ? null : Short.valueOf(stringValue);
        } else if (type == Integer.TYPE) {
            value = zeroValue ? Integer.valueOf(ZERO_VALUE_INT) : Integer.valueOf(stringValue);
        } else if (type == Integer.class) {
            value = zeroValue ? null : Integer.valueOf(stringValue);
        } else if (type == Long.TYPE) {
            value = zeroValue ? Long.valueOf(ZERO_VALUE_LONG) : Long.valueOf(stringValue);
        } else if (type == Long.class) {
            value = zeroValue ? null : Long.valueOf(stringValue);
        } else if (type == Character.TYPE) {
            value = zeroValue ? Character.valueOf(ZERO_VALUE_CHAR) : Character.valueOf(stringValue.charAt(0));
        } else if (type == Character.class) {
            value = zeroValue ? null : Character.valueOf(stringValue.charAt(0));
        } else if (type == Double.TYPE) {
            value = zeroValue ? new Double(ZERO_VALUE_DOUBLE) : new Double(stringValue);
        } else if (type == Double.class) {
            value = zeroValue ? null : new Double(stringValue);
        } else if (type == Float.TYPE) {
            value = zeroValue ? new Float(ZERO_VALUE_FLOAT) : new Float(stringValue);
        } else if (type == Float.class) {
            value = zeroValue ? null : new Float(stringValue);
        } else if (type == Boolean.TYPE) {
            value = zeroValue ? Boolean.valueOf(ZERO_VALUE_BOOLEAN) : Boolean.valueOf(stringValue);
        } else if (type == Boolean.class) {
            value = zeroValue ? null : Boolean.valueOf(stringValue);
        } else if (Date.class.isAssignableFrom(type)) {
            value = zeroValue ? null : TypeTool.parse(stringValue);
        } else if (type == Numeric.class) {
            value = zeroValue ? null : new Numeric(stringValue, 0);
        } else if (type == BigDecimal.class) {
        	value = zeroValue ? null : new BigDecimal(stringValue);
        } else if (type == String.class) {
            value = stringValue;
        } else if (G9Enumerator.class.isAssignableFrom(type) || type.isEnum()) {
            try {
                if (!zeroValue) {
                	value = EnumUtil.getEnumObject(type.getName(), stringValue);
                }
                // else value remains null
            } catch (G9BaseException ex) {
                Object[] msgArgs = { "G9Proxy", type, ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_CREATE_DOMAINOBJ_INSTANCE,
                        msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
        return value;
    }

    /**
     * Sets an attribute using bean-style set method - exept for enumerators,
     * which set using the setCurrentValue-method.
     *
     * @param attributeName the name of the attribute
     * @param value object holding the new value of the attribute
     */
    private void setAttributeUsingMethod(String attributeName, Object value) {
        String methodName = TypeTool.asBeanSetter(attributeName);
        Method setMethod = null;

        // Get reference to the set method...
        // Since we dont know the type of the argument, we need to match name
        Method[] allMethods = domainClass.getMethods();
        for (int i = 0; i < allMethods.length; i++) {
            if (allMethods[i].getName().equals(methodName)) {
                setMethod = allMethods[i];
                break;
            }
        }
        if (setMethod == null) {
            throw new G9BaseException(
                    "Could not find set method for attribute " + attributeName);
        }
        if (setMethod.getParameterTypes().length != 1) {
            throw new G9BaseException("" +
            		"Set method for attribute " + attributeName +
            		" has icorrect signature");
        }

        // Determine type and convert value...
        Class<?> paramType = setMethod.getParameterTypes()[0];
        if (paramType != String.class && (value instanceof String || value == null)) {
            value = convertStringToObjectType((String) value, paramType);
        }
        assert (paramType != null);
        Exception ex = null;
        try {
            // Fixes problem with missing support for model type long.
            if (value != null) {
				if (value.getClass().equals(Integer.class) && paramType.equals(Long.class)) {
					value = Long.valueOf(((Integer) value).longValue());
				}
				// additional special handling of types can be added here
			}
            // Work-around for bug #4533479 in java
            setMethod.setAccessible(true);
            setMethod.invoke(domainObject, new Object[] {value});
        } catch (IllegalArgumentException e) {
            ex = e;
        } catch (IllegalAccessException e) {
            ex = e;
        } catch (InvocationTargetException e) {
            ex = e;
        }
        if (ex != null) {
            Object[] msgArgs = { "G9Proxy", methodName, domainClass, ex.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                    msgArgs);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(ex, msg);
        }
    }

    /**
     * Sets an attribute using direct field access.
     *
     * @param attributeName the name of the attribute
     * @param value object holding the new value of the attribute
     */
    private void setAttributeDirect(String attributeName, Object value) {
        Exception ex = null;
        try {
            Field f = domainClass.getField(attributeName);
            // Work-around for bug #4533479 in java
            f.setAccessible(true);
            f.set(domainObject, value);
        } catch (IllegalAccessException e) {
            ex = e;
        } catch (NoSuchFieldException e) {
            ex = e;
        }
        if (ex != null) {
            Object[] msgArgs = { "G9Proxy", attributeName,
                    domainClass, ex.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                    msgArgs);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(ex, msg);
        }
    }

    /**
     * Relies on reflection to perform the cloning. Reflection can result in a
     * lot of expetions - most of little interest to us in cloning context.
     *
     * @return a cloned instance of the domain object being proxied.
     */
    private T createClone() {
        T clone = null;
        Exception ex = null;
        try {
            clone = domainClass.newInstance();
        } catch (InstantiationException e) {
            ex = e;
        } catch (IllegalAccessException e) {
            ex = e;
        }
        if (ex != null) {
            Object[] msgArgs = { "G9Proxy", domainClass, ex.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_CREATE_DOMAINOBJ_INSTANCE,
                    msgArgs);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(msg);
        }

        Field[] allFields = domainClass.getFields();
        for (int i = 0; i < allFields.length; i++) {
            Field field = allFields[i];
            try {
                // Work-around for bug #4533479 in java
                field.setAccessible(true);
                field.set(clone, field.get(domainObject));
            } catch (IllegalArgumentException e) {
                ex = e;
            } catch (IllegalAccessException e) {
                ex = e;
            }
            if (ex != null) {
                Object[] msgArgs = { "G9Proxy", field.getName(),
                        domainClass, ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                        msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }

        Method[] allMethods = domainClass.getMethods();
        for (int i = 0; i < allMethods.length; i++) {
            String methodName = allMethods[i].getName();
            if (methodName.startsWith("set")) {
                try {
                    Method getMethod = domainClass.getMethod("get"
                            + methodName.substring(3), (Class[]) null);
                    // Work-around for bug #4533479 in java
                    allMethods[i].setAccessible(true);
                    getMethod.setAccessible(true);
                    Object[] args = {getMethod.invoke(domainObject, (Object[]) null)};
                    allMethods[i].invoke(clone, args);
                } catch (IllegalArgumentException e) {
                    ex = e;
                } catch (IllegalAccessException e) {
                    ex = e;
                } catch (InvocationTargetException e) {
                    ex = e;
                } catch (SecurityException e) {
                    ex = e;
                } catch (NoSuchMethodException e) {
                    ex = e;
                }
                if (ex != null) {
                    Object[] msgArgs = { "G9Proxy", methodName,
                            domainClass, ex.getMessage() };
                    Message msg = MessageSystem.getMessageFactory().getMessage(
                            CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                            msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        }

        return clone;
    }

}