/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.exception;

/** A general database exception */
public class DBException extends GeneralException {
    
    private String dbname;

    private String username;

    private String host;

    /**
     * Construcuts a new GeneralException. When the constructor is run, the
     * method log() is called.
     * 
     * @param errorCode The error code for the exception.
     * @param errorMessage The error message for the exception. This message is
     *            to be displayed to the user, so it should be user-friendly.
     * @param subsystem The subsystem of where the exception originated.
     * @param shouldBeLogged Flags if the exception should be logged.
     * @param seriousness The seriousness of the exception. Should be one of
     *            WARNING, ERROR or FATAL_ERROR.
     * @param originalException The exception leading to this exception, if any.
     * @param host The host of where the database resides.
     * @param dbname The database name where the error situation occured.
     * @param username The user name used to connect, if available.
     */
    public DBException(int errorCode, String errorMessage, int subsystem,
            boolean shouldBeLogged, int seriousness,
            Throwable originalException, String host, String dbname,
            String username) {
        super(errorCode, errorMessage, subsystem, shouldBeLogged,
                seriousness, originalException);
        this.dbname = dbname;
        this.username = username;
        this.host = host;
    }

    /**
     * Default empty constructor.
     */
    public DBException() {
        // Empty
    }

    @Override
    protected void log() {
        if (!logException || exceptionIsLogged) {
            return;
        }
        System.out.println(toString());
        exceptionIsLogged = true;
    }

    
    @Override
    public String toString() {
        String superMessage = super.toStringSuper();
        String message = "Database exception - "
                + exceptionSeriousnessToString() + " - CODE:" + errorCode
                + ":" + errorMessage + "\n" + "Host:" + host + " DBName:"
                + dbname + " Username:" + username + "\n" + superMessage;
        if (originalException != null) {
            message += "\nOrignal exception:"
                    + originalException.toString();
        }
        return message;
    }
}
