/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.util.HashMap;
import java.util.Map;

/**
 * Enumeration of valid message replies.
 */
public class MessageReplyType extends MessageEnums implements MessageReply {

    /** Keeps list of created values. */
    private final static Map<Integer, MessageReplyType> values =
        new HashMap<Integer, MessageReplyType>();

    /**
     * Constructor for internal use only.
     *
     * @param pCode Numeric value.
     * @param pName Textual value.
     */
    protected MessageReplyType(final int pCode, final String pName) {
        super(pCode, pName);
        values.put(this.code, this);
    }

    /**
     * Returns the constant corresponding to the given integer, or null if none
     * found..
     *
     * @param pCode Integer value.
     * @return Corresponding constant.
     */
    public static MessageReplyType fromInt(final int pCode) {
        return values.get(Integer.valueOf(pCode));
    }

    /** Reply: No */
    public static final MessageReplyType REPLY_NO = new MessageReplyType(-1,
            getMessageText(CRuntimeMsg.CM_TEXT_NO));

    /** Reply: Cancel */
    public static final MessageReplyType REPLY_CANCEL = new MessageReplyType(0,
            getMessageText(CRuntimeMsg.CM_TEXT_CANCEL));

    /** Reply: OK */
    public static final MessageReplyType REPLY_OK = new MessageReplyType(1,
            getMessageText(CRuntimeMsg.CM_TEXT_OK));

    /** Reply: Yes, equivalent with OK */
    public static final MessageReplyType REPLY_YES = new MessageReplyType(1,
            getMessageText(CRuntimeMsg.CM_TEXT_YES));

    /** Reply: Info */
    public static final MessageReplyType REPLY_INFO = new MessageReplyType(2,
            getMessageText(CRuntimeMsg.CM_TEXT_INFO));

    /** Reply: Ignore */
    public static final MessageReplyType REPLY_IGNORE = new MessageReplyType(3,
            getMessageText(CRuntimeMsg.CM_TEXT_IGNORE));

    /** Reply: Retry */
    public static final MessageReplyType REPLY_RETRY = new MessageReplyType(4,
            getMessageText(CRuntimeMsg.CM_TEXT_RETRY));

    /** Reply: Abort */
    public static final MessageReplyType REPLY_ABORT = new MessageReplyType(5,
            getMessageText(CRuntimeMsg.CM_TEXT_ABORT));

    /** Reply: None specified */
    public static final MessageReplyType REPLY_NONE = new MessageReplyType( Integer.MAX_VALUE,
            getMessageText(CRuntimeMsg.CM_TEXT_UNKNOWN));

    private static String getMessageText(String msgID) {
        Message msg = MessageSystem.getMessageFactory().getMessage(msgID);
        return msg.getMessageText();
    }

}
