/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.convert;


/**
 * Common interface for converters between a model element and a target element.
 * 
 * <p>
 * Note that it is not required to preserve data through the conversion.
 * Meaning, a value object that goes through a round trip of conversion will not
 * necessary result in the original value.
 * 
 * <p>
 * It is generally recommended that converters convert null to null.
 * 
 * <p>
 * Classes implementing this interface are strongly encouraged to also override
 * equals and hashCode as converters are likely to be used in various
 * collections by the g9 framework.
 * 
 * @param <M>
 *            the model type
 * @param <T>
 *            the target type
 */
public interface Converter<M, T> {

	/**
	 * Convert a value object from domain to target.
	 * 
	 * @param value
	 *            The value object to convert
	 * @param context
	 *            The conversion context
	 * @return the converted value
	 * @throws ConvertException
	 *             If conversion can not be performed
	 */
	public T fromModel(M value, ConvertContext context) throws ConvertException;

	/**
	 * Convert a value object from some representation to domain value type.
	 * 
	 * @param value
	 *            The value to convert
	 * @param context
	 *            The conversion context
	 * @return the converted value
	 * @throws ConvertException
	 *             If conversion can not be performed
	 */
	public M toModel(T value, ConvertContext context) throws ConvertException;

}