/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.convert;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.TimeZone;

import no.esito.log.Logger;
import no.esito.util.DateUtil;
import no.g9.exception.SimpleDateConverterException;
import no.g9.support.FormatHelper;

/**
 * SimpleDateConverter is a custom DateTimeConverter which handles g9
 * display rules in a Jouteur and Jasper context.
 *
 * <p>
 * The <code>getAsObject()</code> method parses a String into a
 * <code>java.util.Date</code>, according to the following algorithm:
 * </p>
 * <ul>
 * <li>If the specified String is null, return a <code>null</code>. Otherwise,
 * trim leading and trailing whitespace before proceeding.</li>
 * <li>If the specified String - after trimming - has a zero length, return
 * <code>null</code>.</li>
 * <li>If a <code>displayRule</code> has been specified, its syntax must conform
 * the rules specified by <code>java.text.SimpleDateFormat</code>. Such a
 * displayRule will be used to parse, and the <code>type</code>,
 * <code>dateStyle</code>, and <code>timeStyle</code> properties will be
 * ignored.</li>
 * <li>If a <code>displayRule</code> has not been specified, parsing will be
 * based on the <code>type</code> property, which expects a date value, a time
 * value, or both. Any date and time values included will be parsed in
 * accordance to the styles specified by <code>dateStyle</code> and
 * <code>timeStyle</code>, respectively.</li>
 * <li>In all cases, parsing must be non-lenient; the given string must strictly
 * adhere to the parsing format.</li>
 * </ul>
 *
 * <p>
 * The <code>getAsString()</code> method expects a value of type
 * <code>java.util.Date</code> (or a subclass), and creates a formatted String
 * according to the following algorithm:
 * </p>
 * <ul>
 * <li>If the specified value is null, return a zero-length String.</li>
 * <li>If the specified value is a String, return it unmodified.</li>
 * <li>If a <code>displayRule</code> has been specified, its syntax must conform
 * the rules specified by <code>java.text.SimpleDateFormat</code>. Such a
 * displayRule will be used to format, and the <code>dataType</code>,
 * <code>dateStyle</code>, and <code>timeStyle</code> properties will be
 * ignored.</li>
 * <li>If a <code>displayRule</code> has not been specified, formatting will be
 * based on the <code>dataType</code> property, which includes a date value, a
 * time value, or both into the formatted String. Any date and time values
 * included will be formatted in accordance to the styles specified by
 * <code>dateStyle</code> and <code>timeStyle</code>, respectively.</li>
 * </ul>
 */
public class SimpleDateConverter {
    
    private String dataType;
    private String displayRule;
    private String defaultStyle;
    private Locale locale;


    private static final Logger log = Logger.getLogger(SimpleDateConverter.class);

    /**
     * Create a new g9 date converter.
     * Sets the converter time zone to the current default time zone.
     * @param dataType to be converted
     * @param displayRule for the field
     * @param locale for the converter
     */
    public SimpleDateConverter(final String dataType, final String displayRule, final Locale locale) {
        this.dataType= dataType;
        this.displayRule= displayRule;
        this.defaultStyle= "default";
        this.locale= locale;
    }
    
    /**
     * Create a new g9 date converter.
     * Sets the converter time zone to the current default time zone.
     * Locale is set to system default locale.
     * @param dataType to be converted
     * @param displayRule for the field
     */
    public SimpleDateConverter(final String dataType, final String displayRule) {
        this.dataType= dataType;
        this.displayRule= displayRule;
        this.defaultStyle= "default";
        this.locale= Locale.getDefault();
    }

    /**
     * Convert the specified string value from a field, into a model data object
     * that is appropriate for being stored during the Apply Request Values
     * phase of the request processing life cycle.
     * 
     * @param value to be converted
     * @return value as Date object
     */
    public Object getAsObject(String value) {
    	
        if (log.isTraceEnabled()) {
            log.trace("Parsing value \"" + value + "\", display rule \"" + getDisplayRule()
                    + "\", data type \"" + getDataType() + "\"");
        }

        if (value == null) return null;
        value = value.trim();
        if (value.length() < 1) return null;

        Locale locale = getLocale();
        DateFormat parser = null;
        try {
            if (getDataType().equals("date") || getDataType().equals("timestamp")) {
                parser = parseLocaleFormat(value);
                if (parser == null) {
                    parser = DateUtil.getParseFormat(value, locale);
                    if (parser == null) {
                        parser = getDateFormatFromDisplayRule(locale);
                    }
                }
                if (parser == null) {
                    throw new ParseException("", 0);
                }

            } else {
                parser = getDateFormatFromDisplayRule(locale);
            }
            if (null != getTimeZone()) {
                parser.setTimeZone(getTimeZone());
            }
            return parser.parse(value);
        } catch (Exception e) {
            throw new SimpleDateConverterException(e);
        }
    }

    private Locale getLocale() {
        return this.locale;
    }
    
    private TimeZone getTimeZone() {
        return TimeZone.getDefault();
    }

    
    /**
     * Parses the locale format.
     *
     * @param value the value.
     * @return the date format.
     */
    private DateFormat parseLocaleFormat(String value) {
        DateFormat localeDateFormat = (DateFormat.getDateInstance(DateFormat.MEDIUM, getLocale()));
        localeDateFormat.setLenient(false);
        try {
            localeDateFormat.parse(value);
        } catch (ParseException e) {
            localeDateFormat = null;
        }
        return localeDateFormat;
    }

    
    /**
     * @param value to convert
     * @return String representation of the date
     */
    public String getAsString(Object value) {
        if (value == null)
            return "";
        if (value instanceof String)
            return (String) value;
        return dateToString(value, getLocale());
    }

    /**
     * Converts a date to a string according to this converter.
     *
     * @param value The value
     * @param locale The locale
     * @return the string representation of the date
     */
    String dateToString(Object value, Locale locale) {
        DateFormat formatter = getDateFormatFromDisplayRule(locale);
        if (null != getTimeZone()) {
            formatter.setTimeZone(getTimeZone());
        }
        return formatter.format(value);
    }

    /**
     * Gets the date format from the displayrule.
     *
     * @param locale the locale
     * @return the date format
     */
    DateFormat getDateFormatFromDisplayRule(Locale locale) {
        if (getDisplayRule() == null && getDataType() == null) {
            return null;
        }

        DateFormat df = null;
        if (getDisplayRule() != null && !getDisplayRule().equals("")) {
            String format = FormatHelper.getDatetimeFormat(getDisplayRule());
            if (format == null) {
                throw new IllegalArgumentException("Unsupported display rule \"" + getDisplayRule() + "\"");
            }
            if (log.isTraceEnabled()) {
                log.trace("Using format \"" + format + "\" for display rule \"" + getDisplayRule() + "\"");
            }
            df = new SimpleDateFormat(format, locale);
        } else if (getDataType().equals("both") || getDataType().equals("timestamp")) {
            df = DateFormat.getDateTimeInstance(getStyle(getDefaultStyle()), getStyle(getDefaultStyle()),
                    locale);
        } else if (getDataType().equals("date")) {
            df = DateFormat.getDateInstance(getStyle(getDefaultStyle()), locale);
        } else if (getDataType().equals("time")) {
            df = DateFormat.getTimeInstance(getStyle(getDefaultStyle()), locale);
        } else {
            throw new IllegalArgumentException("Invalid datatype: " + getDataType());
        }
        df.setLenient(false);
        return df;
    }
    
    
    /**
     * Return the style constant for the specified style name.
     *
     * @param name Name of the style for which to return a constant
     * @throws SimpleDateConverterException if the style name is not valid
     */
    private int getStyle(String name) {

        if ("default".equals(name)) {
            return (DateFormat.DEFAULT);
        } else if ("short".equals(name)) {
            return (DateFormat.SHORT);
        } else if ("medium".equals(name)) {
            return (DateFormat.MEDIUM);
        } else if ("long".equals(name)) {
            return (DateFormat.LONG);
        } else if ("full".equals(name)) {
            return (DateFormat.FULL);
        } else {
            throw new SimpleDateConverterException("Invalid style '" + name + '\'');
        }

    }

    /**
     * <p>
     * Set the format displayRule to be used when formatting and parsing dates
     * and times. Valid values are those supported by
     * <code>java.text.SimpleDateFormat</code>. An invalid value will cause a
     * {@link SimpleDateConverterException} when <code>getAsObject()</code> or
     * <code>getAsString()</code> is called.
     * </p>
     *
     * @param displayRule The new format displayRule
     */
    public void setDisplayRule(String displayRule) {
        this.displayRule= displayRule;
    }

    
    /**
     * @return displayRule
     */
    public String getDisplayRule() {
        return this.displayRule;
    }
    
    
    /**
     * @param dataType to be set
     */
    public void setDataType(String dataType) {
        this.dataType= dataType;
    }

    
    /**
     * @return dataType
     */
    public String getDataType() {
        return this.dataType;
    }


    /**
     * @return defaultStyle
     */
    public String getDefaultStyle() {
        return this.defaultStyle;
    }
    
    /**
     * @param defaultStyle to be set
     */
    public void setDefaultStyle(final String defaultStyle) {
        this.defaultStyle= defaultStyle;
    }
}
