/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.log;

/**
 * A wrapper for the log4j logger.
 * 
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 * 
 */
public final class Logger {

    /**
     * The delegate that performs the logging.
     */
    private final org.apache.logging.log4j.Logger delegate;

    /**
     * Constructs a new logger that will use the specified delegate to log.
     * 
     * @param delegate the log delegate.
     */
    private Logger(org.apache.logging.log4j.Logger delegate) {
        this.delegate = delegate;
    }

    /**
     * Get the specified logger.
     * 
     * @param name The name of the logger to retrieve.
     * @return the specified logger.
     */
    public static Logger getLogger(String name) {
        return new Logger(org.apache.logging.log4j.LogManager.getLogger(name));
    }

    /**
     * Shorthand for <code>getLogger(clazz.getName())</code>.
     * 
     * @param clazz The name of <code>clazz</code> will be used as the name of
     *            the logger to retrieve. See {@link #getLogger(String)}.
     * @return the specified logger.
     */
    public static Logger getLogger(Class<?> clazz) {
        return new Logger(org.apache.logging.log4j.LogManager.getLogger(clazz));
    }

    /**
     * Log a trace message.
     * <p>
     * <b>WARNING</b> Note that passing a {@link Throwable} to this method will
     * print the name of the <code>Throwable</code> but no stack trace. To print
     * a stack trace use the {@link #trace(Object, Throwable)} form instead.
     * 
     * @param message the message object to log.
     */
    public void trace(Object message) {
        delegate.trace(message);
    }

    /**
     * Log a trace message including the stack trace.
     * 
     * @param message the message object to log
     * @param t the throwable to log, including its stack trace.
     */
    public void trace(Object message, Throwable t) {
        delegate.trace(message, t);
    }

    /**
     * Log a debug message.
     * <p>
     * <b>WARNING</b> Note that passing a {@link Throwable} to this method will
     * print the name of the <code>Throwable</code> but no stack trace. To print
     * a stack trace use the {@link #debug(Object, Throwable)} form instead.
     * 
     * @param message the message object to log.
     */
    public void debug(Object message) {
        delegate.debug(message);
    }

    /**
     * Log a debug message including the stack trace.
     * 
     * @param message the message object to log.
     * @param t the exception to log, including its stack trace.
     */
    public void debug(Object message, Throwable t) {
        delegate.debug(message, t);
    }

    /**
     * Log an error message.
     * <p>
     * <b>WARNING</b> Note that passing a {@link Throwable} to this method will
     * print the name of the <code>Throwable</code> but no stack trace. To print
     * a stack trace use the {@link #debug(Object, Throwable)} form instead.
     * 
     * @param message the message object to log.
     */
    public void error(Object message) {
        delegate.error(message);
    }

    /**
     * Log an error message including the stack trace.
     * 
     * @param message the message object to log.
     * @param t the exception to log, including its stack trace.
     */
    public void error(Object message, Throwable t) {
        delegate.error(message, t);
    }

    /**
     * Log a fatal message.
     * <p>
     * <b>WARNING</b> Note that passing a {@link Throwable} to this method will
     * print the name of the <code>Throwable</code> but no stack trace. To print
     * a stack trace use the {@link #debug(Object, Throwable)} form instead.
     * 
     * @param message the message object to log.
     */
    public void fatal(Object message) {
        delegate.fatal(message);
    }

    /**
     * Log a fatal message including the stack trace.
     * 
     * @param message the message object to log.
     * @param t the exception to log, including its stack trace.
     */
    public void fatal(Object message, Throwable t) {
        delegate.fatal(message, t);
    }

    /**
     * Log an info message.
     * <p>
     * <b>WARNING</b> Note that passing a {@link Throwable} to this method will
     * print the name of the <code>Throwable</code> but no stack trace. To print
     * a stack trace use the {@link #debug(Object, Throwable)} form instead.
     * 
     * @param message the message object to log.
     */
    public void info(Object message) {
        delegate.info(message);
    }

    /**
     * Log an info message including the stack trace.
     * 
     * @param message the message object to log.
     * @param t the exception to log, including its stack trace.
     */
    public void info(Object message, Throwable t) {
        delegate.info(message, t);
    }

    /**
     * Check whether this category is enabled for the TRACE Level.
     * 
     * @return boolean - <code>true</code> if this category is enabled for level
     *         TRACE, <code>false</code> otherwise.
     */
    public boolean isTraceEnabled() {
        return delegate.isTraceEnabled();
    }

    /**
     * Check whether this category is enabled for the <code>DEBUG</code> Level.
     * <p>
     * If you are worried about speed, then you should write
     * 
     * <pre>
     * if (cat.isDebugEnabled()) {
     *     cat.debug(&quot;This is entry number: &quot; + i);
     * }
     * </pre>
     * <p>
     * This way you will not incur the cost of parameter construction if
     * debugging is disabled for <code>cat</code>. On the other hand, if the
     * <code>cat</code> is debug enabled, you will incur the cost of evaluating
     * whether the category is debug enabled twice. Once in
     * <code>isDebugEnabled</code> and once in the <code>debug</code>. This is
     * an insignificant overhead since evaluating a category takes about 1%% of
     * the time it takes to actually log.
     * 
     * @return boolean - <code>true</code> if this category is debug enabled,
     *         <code>false</code> otherwise.
     */
    public boolean isDebugEnabled() {
        return delegate.isDebugEnabled();
    }

    /**
     * Check whether this category is enabled for the info Level. See also
     * {@link #isDebugEnabled}.
     * 
     * @return boolean - <code>true</code> if this category is enabled for level
     *         info, <code>false</code> otherwise.
     */
    public boolean isInfoEnabled() {
        return delegate.isInfoEnabled();
    }

    /**
     * Log a warn message.
     * <p>
     * <b>WARNING</b> Note that passing a {@link Throwable} to this method will
     * print the name of the <code>Throwable</code> but no stack trace. To print
     * a stack trace use the {@link #debug(Object, Throwable)} form instead.
     * 
     * @param message the message object to log.
     */
    public void warn(Object message) {
        delegate.warn(message);
    }

    /**
     * Log a warn message including the stack trace.
     * 
     * @param message the message object to log.
     * @param t the exception to log, including its stack trace.
     */
    public void warn(Object message, Throwable t) {
        delegate.warn(message, t);
    }

    /**
     * Omits a trace message with the invoking method name and line number. More
     * specifically - the message is:
     * <code>"Barking from method &lt;method name&gt;, at line: &lt;line number&gt;."</code>
     * where "method name" is the name of the method where <code>bark</code> is
     * invoked
     */
    public void bark() {
        if (isTraceEnabled()) {
            String invokerNameAndLineNumber = getInvokerNameAndLineNumber();
            trace(invokerNameAndLineNumber);
        }
    }

    /**
     * Returns the name of the invoking method.
     * 
     * @return the invoking method name.
     */
    private String getInvokerNameAndLineNumber() {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
        for (int i = 2; i < stackTrace.length; i++) {
            StackTraceElement stackTraceElement = stackTrace[i];
            if (!stackTraceElement.getClassName().equals(getClass().getName())) {
                String s =
                        "Barking from method "
                                + stackTraceElement.getMethodName();
                s += ", at line:" + stackTraceElement.getLineNumber() + ".";
                return s;
            }
        }
        return null;
    }

    /**
     * Creates a string useful for logging parameters to a method. The passed-in
     * parameters array should contain pairs of information, typically parameter
     * name -> parameter value, e.g. {"name", name, "age", age}. Assuming Donald
     * as name value and 42 as age value, this method would produce the
     * following String:
     * 
     * <pre>
     * &quot;name [Donald], age [42]&quot;
     * </pre>
     * 
     * Typical usage:
     * 
     * <pre>
     * foo(String name, int age) {
     *    String s = Logger.formatParameters(new Object[] {"name", name, 
     *                                                     "age", age});
     *    log.trace("invoked with the following values" + s);
     *    ...
     * }
     * </pre>
     * 
     * @param parameters paired object array of parameter name and value
     * @return a
     */
    public static String formatParameters(Object[] parameters) {
        int i = 0;
        StringBuffer sb = new StringBuffer();
        while (i < parameters.length) {
            String paramName = null;
            String paramValue = null;
            paramName = String.valueOf(parameters[i]);
            i++;
            if (i < parameters.length) {
                paramValue = String.valueOf(parameters[i]);
                i++;
            }
            sb.append(paramName + " [" + paramValue + "]" + ", ");

        }
        return sb.substring(0, sb.length() - 2);
    }

}
