/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.util;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

/**
 * The Class DateUtil.
 */
public class DateUtil {
	private static final Calendar CALENDAR = Calendar.getInstance();
	/** The supported yyyy formats. */
	private static String[] supportedYYYYFormats = load4YDateFormats();
	/** The supported yy formats. */
	private static String[] supportedYYFormats = load2YDateFormats();
	/** The supported mmm formats. */
	private static String[] supportedMMMFormats = load3MDateFormats();
	/**
	 * Load yyyy date formats.
	 *
	 * @return the string[]
	 */
	private static String[] load4YDateFormats() {
		return new String[]{"dd.MM.yyyy",
				"dd/MM/yyyy",
				"dd-MM-yyyy",
				"dd.MM.yyyy HH",
				"dd/MM/yyyy HH",
				"dd-MM-yyyy HH",
				"dd.MM.yyyy HH:mm",
				"dd/MM/yyyy HH:mm",
				"dd-MM-yyyy HH:mm",
				"dd.MM.yyyy HH:mm:ss",
				"dd/MM/yyyy HH:mm:ss",
				"dd-MM-yyyy HH:mm:ss"};
	}
	/**
	 * Load yy date formats.
	 *
	 * @return the string[]
	 */
	private static String[] load2YDateFormats() {
		String[] temp = { "dd.MM.yy", 
		        "dd/MM/yy", 
		        "dd-MM-yy" };
		return temp;
	}
	/**
	 * Load MMM date formats.
	 *
	 * @return the string[]
	 */
	private static String[] load3MDateFormats() {
		String[] temp = { "dd.MMM.yyyy", 
		        "dd/MMM/yyyy",
				"dd-MMM-yyyy"};
		return temp;
	}
	// -----------------------------------------------------------------------
	/**
	 * Gets the parses the format.
	 *
	 * @param inputDateStr
	 *            the input date str
	 * @param locale
	 *            the locale
	 * @return the parses the format
	 */
    public static DateFormat getParseFormat(String inputDateStr, Locale locale) {
        if (!isValidDate(inputDateStr))
            return null;
        String[] formats = getFormats(inputDateStr);
        DateFormat formatRef;
        // iterate over the array and parse
        for (int i = 0; i < formats.length; i++) {
            try {
                formatRef = new SimpleDateFormat(formats[i], locale);
                formatRef.setLenient(false);
                Date d = formatRef.parse(inputDateStr);
                if (!isInteger(inputDateStr)) {
                    if (formatRef.format(d).equals(inputDateStr)) {
                        return formatRef;
                    }
                }
                else {
                    return formatRef;
                }
            } catch (Exception e) {
                // do nothing because we want to try the next
                // format if current one fails
            }
        }
        // haven't returned so couldn't parse
        return null;
    }

	/**
	 * Gets the formats.
	 *
	 * @param inputDateStr
	 *            the input date str
	 * @return the formats
	 */
	private static String[] getFormats(String inputDateStr) {
		if (isInteger(inputDateStr)) {
			switch (inputDateStr.length()) {
			case 8:
				return new String[]{"ddMMyyyy"};
			case 10:
				return new String[]{"ddMMyyyyHH"};
			case 12:
				return new String[]{"ddMMyyyyHHmm"};
			default:
				return new String[]{"ddMMyyyyHHmmss"};
			}
		}
        if (inputDateStr.length() == 8) {
        	return supportedYYFormats;
        }
        if (inputDateStr.length() == 11) {
        	return supportedMMMFormats;
        }
        return supportedYYYYFormats;
	}
	/**
	 * Checks if the given input is Integer.
	 *
	 * @param input
	 *            the input
	 * @return true, if is integer
	 */
	public static boolean isInteger(String input) {
        for (int i = 0; i < input.length(); i++) {
            //If we find a non-digit character we return false.
            if (!Character.isDigit(input.charAt(i)))
                return false;
        }
        return true;
	}
	/**
	 * Checks if the given value is valid date.
	 *
	 * @param value
	 *            the value
	 * @return true, if is valid date
	 */
    public static boolean isValidDate(String value) {
        if (value == null || value.length() < 8) {
			return false;
		}
		return true;
	}
	/**
	 * End of day.
	 *
	 * @param date
	 *            the date
	 * @return the date
	 */
	public static Date endOfDay(Date date) {
		Calendar calendar = CALENDAR;
		synchronized (calendar) {
			calendar.setTime(date);
			calendar.set(Calendar.HOUR_OF_DAY, 23);
			calendar.set(Calendar.MILLISECOND, 999);
			calendar.set(Calendar.SECOND, 59);
			calendar.set(Calendar.MINUTE, 59);
			return calendar.getTime();
		}
	}

	/**
	 * Start of day.
	 *
	 * @param date
	 *            the date
	 * @return the date
	 */
	public static Date startOfDay(Date date) {
		Calendar calendar = CALENDAR;
		synchronized (calendar) {
			calendar.setTime(date);
			calendar.set(Calendar.HOUR_OF_DAY, 0);
			calendar.set(Calendar.MILLISECOND, 0);
			calendar.set(Calendar.SECOND, 0);
			calendar.set(Calendar.MINUTE, 0);
			return calendar.getTime();
		}
	}
	/**
	 * String to date.
	 *
	 * @param str_date
	 *            the String value
	 * @param format
	 *            the format
	 * @return the date
	 */
	public static Date stringToDate(String str_date, String format) {
		DateFormat formatter;
		Date date = null;
		try {
			formatter = new SimpleDateFormat(format);
			date = formatter.parse(str_date);
		} catch (ParseException e) {
		    // do nothing because date will be returned as null
		}
		return date;
	}

	/**
	 * Date to string.
	 *
	 * @param date the date value to be converted to String
	 * @param format the format
	 * @return the string
	 */
	public static String dateToString(Date date, String format) {
		DateFormat formatter;
		String str_date = null;
		formatter = new SimpleDateFormat(format);
		str_date = formatter.format(date);
		return str_date;
	}
}

