/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.util;

import java.lang.reflect.Method;

import no.g9.exception.G9BaseException;
import no.g9.support.G9Enumerator;
import no.g9.support.ObjectFactory;

/**
 * Helper methods for enums.
 */
public class EnumUtil {

    /**
     * Get the enum value object for the given value.
     * The method handles both Java style enums and G9Enumerator.
     *
     * @param <T> the return type
     * @param enumClassName the class name for the enumerator
     * @param value the integer value (as given in the class model for the enum)
     * @return the enum value object
     */
    @SuppressWarnings("unchecked")
	public static <T> T getEnumObject(String enumClassName, int value) {
    	Object enumValue = null;
    	Class<?> enumClass = null;
    	try {
			enumClass = Class.forName(enumClassName);
			if (enumClass.isEnum()) {
				Method get = enumClass.getMethod("get", int.class);
				enumValue = get.invoke(null, value);
			}
			else {
            	G9Enumerator geVal = (G9Enumerator)ObjectFactory.newObject(enumClassName);
                geVal.setCurrentValue(value);
                enumValue = geVal;
			}
		} catch (Exception e) {
			throw new G9BaseException("Unable to get enum object", e);
		}
    	return (T) enumValue;
    }

    /**
     * Get the enum value object for the given literal.
     * The method handles both Java style enums and G9Enumerator.
     *
     * @param <T> the return type
     * @param enumClassName the class name for the enumerator
     * @param literal the String literal (as given in the class model for the enum)
     * @return the enum value object
     */
    @SuppressWarnings("unchecked")
	public static <T> T getEnumObject(String enumClassName, String literal) {
    	Object enumValue = null;
    	Class<?> enumClass = null;
    	try {
    		enumClass = Class.forName(enumClassName);
    		if (enumClass.isEnum()) {
    			Method get = enumClass.getMethod("get", String.class);
    			enumValue = get.invoke(null, literal);
    		}
    		else {
    			G9Enumerator geVal = (G9Enumerator)ObjectFactory.newObject(enumClassName);
        		try {
        			Method toValue = geVal.getClass().getMethod("toValue", String.class);
        			int curVal = (int) toValue.invoke(null, literal);
        			geVal.setCurrentValue(curVal);
        			enumValue = geVal;
        		} catch (Exception e) {
        			throw new G9BaseException("Unable to get enum object", e);
        		}
    		}
    	} catch (Exception e) {
    		throw new G9BaseException("Unable to get enum object", e);
    	}
    	return (T) enumValue;
    }
    
    /**
     * Get the enum value for the given enum object.
     * The method handles both Java style enums and G9Enumerator.
     *
     * @param enumObject the enum object to get the value for
     * @return the enum value
     */
    public static int getEnumValue(Object enumObject) {
    	if (enumObject == null) {
    		throw new IllegalArgumentException("Enum value null is not allowed");
    	}
    	if (enumObject instanceof G9Enumerator) {
    		return ((G9Enumerator) enumObject).getCurrentValue();
    	}
    	else {
    		try {
    			Method getValue = enumObject.getClass().getMethod("getValue", (Class<?>[])null);
    			return (int) getValue.invoke(enumObject, (Object[])null);
    		} catch (Exception e) {
    			throw new G9BaseException("Unable to get enum value", e);
    		}
    	}
    }

    /**
     * Get the enum literal for the given enum object.
     * The method handles both Java style enums and G9Enumerator.
     *
     * @param enumObject the enum object to get the literal for
     * @return the enum literal
     */
    public static String getEnumLiteral(Object enumObject) {
    	if (enumObject == null) {
    		throw new IllegalArgumentException("Enum value null is not allowed");
    	}
    	if (enumObject instanceof G9Enumerator) {
    		try {
    			Method toTitle = enumObject.getClass().getMethod("toTitle", int.class);
    			return (String) toTitle.invoke(null, ((G9Enumerator)enumObject).getCurrentValue());
    		} catch (Exception e) {
    			throw new G9BaseException("Unable to get enum literal", e);
    		}
    	}
    	else {
    		try {
    			Method getValue = enumObject.getClass().getMethod("getLiteral", (Class<?>[])null);
    			return (String) getValue.invoke(enumObject, (Object[])null);
    		} catch (Exception e) {
    			throw new G9BaseException("Unable to get enum literal", e);
    		}
    	}
    }

}
