/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.util;

import java.text.ParseException;

import no.g9.support.FormatHelper;
import no.g9.support.G9Consts;
import no.g9.support.Numeric;

/**
 * This class provides support for converting number values to and from strings.
 * The conversion is done according to the given g9 display rule.
 */
public class NumberFormat {

    private int datatype;
    private String displayRule;

    /**
     * Create a new NumberFormat for the given data type and display rule.
     *
     * @param datatype the data type for this formatter.
     * @param displayRule the display rule to be used for the formatting.
     */
    public NumberFormat(int datatype, String displayRule) {
        this.datatype = datatype;
        this.displayRule = displayRule;
    }

    /**
     * Format the given number according to the provided format.
     *
     * @param format the format to use.
     * @param number the value to be formatted.
     * @param internalFormat if true, the result is not dependent on the locale settings.
     *
     * @return a string with the formatted number value.
     */
    public String format(String format, Object number, boolean internalFormat) {

        String inputFormat = getInputFormat();
        int intLength = getIntLength();
        int decimalLength = getDecimalLength();
        int sign = FormatHelper.getSign(displayRule);

        int maxLength = inputFormat.length();

        int sign1 = sign;
        if (format.equals(inputFormat)) {
            if (sign1 == FormatHelper.SIGN_MINUS)
                sign1 = FormatHelper.SIGN_FLOATING_MINUS;
            if (sign1 == FormatHelper.SIGN_PLUS)
                sign1 = FormatHelper.SIGN_FLOATING_PLUS;
        }
        StringBuffer res = new StringBuffer();
        if (number != null) {
            String bStr = number.toString();
            int intPos = 0;
            boolean negative = false;
            if (bStr.length() > 0) {
                if (bStr.length() > 0 && bStr.charAt(intPos) == '-') {
                    negative = true;
                    intPos++;
                }
                char dot = '.';
                int noDigits = bStr.indexOf(dot);
                if (noDigits < 0)
                    noDigits = bStr.length();
                noDigits = noDigits - intPos;
                if (noDigits > intLength
                        || (negative && sign1 == FormatHelper.SIGN_NONE)) {
                    for (noDigits = 0; noDigits < maxLength; noDigits++) {
                        res.append('#');
                    }
                    return res.toString();
                }
                int dotPos = format.indexOf('.');
                int signPos = addInt(dotPos < 0 ? format : format.substring(0,
                        dotPos + 1), sign1, internalFormat, noDigits < 0 ? "0"
                        : bStr.substring(intPos, noDigits + intPos), intLength,
                        res);
                if (dotPos >= 0) {
                    intPos = bStr.indexOf(dot) + 1;
                    if (intPos > 0 && intPos <= bStr.length()) {
                        noDigits = bStr.length() - intPos;
                        bStr = bStr.substring(intPos);
                        while (decimalLength > noDigits) {
                            bStr = bStr.concat("0");
                            noDigits++;
                        }
                    } else {
                        bStr = "";
                    }
                    int tmp = addInt(format.substring(dotPos + 1), sign1,
                            internalFormat, bStr, decimalLength, res);
                    if (signPos == -1)
                        signPos = tmp;
                }
                if (signPos >= 0 && sign1 != FormatHelper.SIGN_NONE) {
                    if (negative) {
                        res.insert(signPos, '-');
                    } else if (sign1 == FormatHelper.SIGN_PLUS
                            || sign1 == FormatHelper.SIGN_FLOATING_PLUS) {
                        res.insert(signPos, '+');
                    } else if (sign1 == FormatHelper.SIGN_MINUS) {
                        res.insert(signPos, ' ');
                    }
                }
                if (format.equals(inputFormat)
                        || format.charAt(0) == '#'
                        || (format.charAt(0) == '+'
                        && sign1 != FormatHelper.SIGN_MINUS && sign1 != FormatHelper.SIGN_PLUS)) {
                    int i = 0;
                    while (i < res.length() && res.charAt(i) == ' ')
                        i++;
                    if (i > 0) {
                        res.delete(0, i);
                    }
                    i = res.length() - 1;
                    while (i >= 0 && res.charAt(i) == ' ')
                        i--;
                    i++;
                    if (i >= 0) {
                        res.delete(i, res.length());
                    }
                }
            }
        }
        return res.toString();
    }

    /**
     * Convert the given string into a number value.
     *
     * @param str the string to parse.
     * @param internalFormat if true, the result is not dependent on the locale settings.
     *
     * @return the number value of the converted string.
     * @throws ParseException if the given string value cannot be parsed.
     */
    public Object parse(String str, boolean internalFormat)
            throws ParseException {
        Object retVal = null;
        if (str != null && str.length() != 0) {

            // Convert string so default parsing can be used
            // I.e. remove all but digits and minus and
            // replace local decimal separator with .
            char decimalSeparator = internalFormat ? '.' : FormatHelper.getDecimalSeparator();
            StringBuffer numBuffer = new StringBuffer();
            StringBuffer decimalBuffer = numBuffer;
            int i;
            for (i = 0; i < str.length(); i++) {
                char c = str.charAt(i);
                if (c == '-' && internalFormat && i != 0) {
                    throw new ParseException("Illegal character", i);
                }
                if (Character.isDigit(c) || c == '-') {
                    decimalBuffer.append(c);
                } else if (c == decimalSeparator) {
                    if (datatype == G9Consts.DT_SHORTINT
                            || datatype == G9Consts.DT_LONGINT
                            || datatype == G9Consts.DT_LONGLONG) {
                        if (internalFormat) {
                            throw new ParseException("Illegal character", i);
                        }
                        decimalBuffer = new StringBuffer();
                    } else {
                        decimalBuffer.append('.');
                    }
                } else if (internalFormat) {
                    throw new ParseException("Illegal character", i);
                }
            }
            int round = 0;
            if (decimalBuffer != numBuffer && decimalBuffer.length() > 0) {
                char c = decimalBuffer.charAt(0);
                if (Character.isDigit(c) && Character.digit(c, 10) >= 5) {
                    round = (numBuffer.length() > 0 && numBuffer.charAt(0) == '-') ? -1
                            : 1;
                }
            }
            try {
                switch (datatype) {
                    case G9Consts.DT_SHORTINT:
                        retVal = Short.valueOf(numBuffer.toString());
                        if (round != 0) {
                            short sh = ((Short) retVal).shortValue();
                            retVal = Short.valueOf((short) (sh + round));
                        }
                        break;
                    case G9Consts.DT_LONGINT:
                        retVal = Integer.valueOf(numBuffer.toString());
                        if (round != 0) {
                            int in = ((Integer) retVal).intValue();
                            retVal = Integer.valueOf(in + round);
                        }
                        break;
                    case G9Consts.DT_LONGLONG:
                        retVal = Long.valueOf(numBuffer.toString());
                        if (round != 0) {
                            long in = ((Long) retVal).longValue();
                            retVal = Long.valueOf(in + round);
                        }
                        break;
                    case G9Consts.DT_NUMERIC:
                        retVal = new Numeric(numBuffer.toString(),
                                getDecimalLength());
                        break;
                    case G9Consts.DT_REAL:
                        retVal = new Float(numBuffer.toString());
                        break;
                    case G9Consts.DT_DOUBLE:
                        retVal = new Double(numBuffer.toString());
                        break;
                    default:
                        break;
                }
            } catch (NumberFormatException e) {
                throw new ParseException(e.getMessage(), 0);
            }
        }
        return retVal;
    }

    /**
     * Check if the given number value is zero or null.
     *
     * @param value the value to check.
     * @return true only if the value is zero or null.
     */
    public static boolean isZero(Object value) {
        boolean result = true;
        if (value != null) {
            if (value instanceof Numeric) {
                result = ((Numeric) value).compareTo(new Numeric(0,
                        ((Numeric) value).getScale())) == 0;
            } else if (value instanceof Short) {
                result = ((Short) value).compareTo(Short.valueOf((short) 0)) == 0;
            } else if (value instanceof Integer) {
                result = ((Integer) value).compareTo(Integer.valueOf(0)) == 0;
            } else if (value instanceof Long) {
                result = ((Long) value).compareTo(Long.valueOf(0)) == 0;
            } else if (value instanceof Float) {
                result = ((Float) value).compareTo(new Float(0.0)) == 0;
            } else if (value instanceof Double) {
                result = ((Double) value).compareTo(new Double(0.0)) == 0;
            }
        }
        return result;
    }

    private int addInt(String format, int sign1, boolean internalFormat,
            String number, int maxLength, StringBuffer buffer) {
        int signPos = -1;
        int resPos = 0;
        int intPos = 0;
        int formatPos = -1;
        int noDigits = number.length();
        for (formatPos = 0; formatPos < format.length(); formatPos++) {
            char c = format.charAt(formatPos);
            switch (c) {
                case '+':
                    signPos = resPos;
                    break;
                case '#':
                case '0':
                    if (noDigits < maxLength) {
                        noDigits++;
                        if (c == '0') {
                            buffer.append('0');
                        } else {
                            buffer.append(' ');
                            if (signPos >= 0
                                    && (sign1 == FormatHelper.SIGN_FLOATING_MINUS || sign1 == FormatHelper.SIGN_FLOATING_PLUS)) {
                                signPos = resPos;
                            }
                        }
                    } else {
                        buffer.append((number.length() == 1 && c == '#' && number.charAt(0) == '0') ? ' ' :
                            number.charAt(intPos));
                        intPos++;
                    }
                    break;
                case ',':
                    buffer.append(FormatHelper.getGroupingSeparator());
                    break;
                case '.':
                    buffer.append(internalFormat ? '.' : FormatHelper.getDecimalSeparator());
                    break;
                default:
                    buffer.append(c);
                    break;
            }
            resPos++;
        }
        return signPos;
    }

    private String getInputFormat() {
        String outputFormat = FormatHelper.getNumericFormat(displayRule);
        return FormatHelper.getNumericInputFormat(outputFormat);
    }

    private int getDecimalLength() {
        int decimalLength = 0;
        String inputFormat = getInputFormat();
        if (inputFormat.indexOf('.') != -1) {
            decimalLength = Math.max(inputFormat.length() - getIntLength() - 1, 0);
        }
        return decimalLength;
    }

    private int getIntLength() {
        int intLength = getInputFormat().indexOf('.');
        if (intLength == -1) {
            intLength = getInputFormat().length();
        }
        int sign = FormatHelper.getSign(displayRule);
        if (sign != FormatHelper.SIGN_NONE) {
            intLength--;
        }
        return intLength;
    }

}
