/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.domain;

/**
 * The domain object proxy interface
 * 
 * @param <T> Type of domain object
 */
public interface DomainObjectProxy<T> {

    /**
     * Quiering about the object state. Returns <code>true</code> if the object
     * state is unchanged from the initial state (in other words clean).
     * 
     * @return <code>true</code> if the proxied object is clean.
     */
    boolean isClean();
    
    /**
     * Quiering about the attribute state. Returns <code>true</code> if the
     * attribute is unchanged from it's original state.
     * 
     * @param attributeName (missing javadoc)
     * @return true if attribute is unchanged
     */
    boolean isClean(String attributeName);

    /**
     * Returns the object in its initial state.
     * 
     * @return a clean object.
     */
    T getCleanVersionOfObject();

    /**
     * Returns the object in its current state, which might be clean, might
     * be dirty.
     * 
     * @return the object in its current state.
     */
    T getObject();

    /**
     * Sets an attribute. If the attribute is sucessfully set, the object state
     * is considered dirty, no matter what the previous state was. If the
     * specified attribute name does not match an actual attribute which is
     * possible to set (either directly or via a set-method), an
     * IllegalArgumentException is thrown.
     * 
     * @param attributeName the name of the attribute to set.
     * @param attributeValue the new value of the specified attribute. Primitive
     *            types must be wrapped.
     */
    void setAttribute(String attributeName, Object attributeValue);

    /**
     * Returns the specified attribute with its current value. Primitives are
     * wrapped. If the specified attribute name does not match an actual
     * attribute which is possible to get (either directly or via a get-method),
     * an IllegalArgumentException is thrown.
     * 
     * @param attributeName the name of the attribute to get.
     * @return the current value of the attribute.
     */
    Object getAttribute(String attributeName);
    
    /**
     * Returns the initial value of the specified attribute.
     * 
     * @param attributeName (missing javadoc)
     * @return the attribute object
     */
    Object getCleanVersionOfAttribute(String attributeName);

    /**
     * Resets the specified attribute to its original value.
     * 
     * @param attributeName the (declared) name of the attribute
     */
    void resetAttribute(String attributeName);
    
    /**
     * Checks if the specified value equals the original value from the domain
     * object. All Strings not containing any letters (including null-pointers)
     * are considered equal. Consider the following: <code>
     * String s1 = null;
     * String s2 = "";
     * String s3 = " ";
     * String orig1 = null;
     * </code>
     * Asuming <code>orig1</code> is a reference to the domain-objects initial
     * value of the attribute, <code>s1</code>, <code>s2</code> and
     * <code>s3</code> are all considered equal to the origianal value.
     * 
     * 
     * @param attributeName the (declared) name of the attribute
     * @param value - object holding the value which sould be compared to the
     *            initial value
     * @return <code>true</code> if the specified value equals the initial
     *         value.
     */
    boolean equalsInitialValue(String attributeName, Object value);
    
    /**
     * Resets the object to its initial, clean state
     * 
     */
    void reset();
    
    /**
     * Forces the specified value to be the clean value.
     * 
     * @param attributeName the name of the attribute.
     * @param value the (clean) value of the specified attribute
     */
    void forceClean(String attributeName, Object value);
    
    /**
     * Copies the status from an attribute in <code>fromProxy</code>.
     * 
     * @param fromProxy (missing javadoc)
     * @param fromAttributeName (missing javadoc)
     * @param toAttributeName (missing javadoc)
     */
    void copyStatus(DomainObjectProxy<?> fromProxy, String fromAttributeName, String toAttributeName); 
    
    
    /**
     * Forces the proxy to hold a new object, which then are considered "clean"
     * 
     * @param o (missing javadoc)
     */
    void forceClean(T o);
        
    
    

}
