/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.domain;

import java.io.Serializable;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import no.g9.exception.G9BaseException;
import no.g9.support.Registry;

import org.hibernate.LazyInitializationException;
import org.hibernate.collection.internal.PersistentSet;

/**
 * Utility class for domain objects.
 */
public class DomainUtil {

    private static final String PROP_IGNORE_LOCKFLAG = "DomainUtil.ignoreLockFlag";
    private static boolean ignoreLockFlag;

    static {
        DomainUtil.ignoreLockFlag= false;
        String prop= Registry.getRegistry().getG9Property(DomainUtil.PROP_IGNORE_LOCKFLAG);
        if ("TRUE".equalsIgnoreCase(prop)) {
            DomainUtil.ignoreLockFlag = true;
        }
    }

    // Tool box class. Prevent instances.
    private DomainUtil() {
        // Empty
    }

    /**
     * Get the "true" domain class for a given domainObject, i.e. avoid
     * Hibernate proxy classes etc.
     *
     * @param <T>
     *            The domain object (super) class
     * @param <U>
     *            The domain object type.
     * @param domainObject
     *            The domain object to get the class from
     * @return The domain object class.
     */
    @SuppressWarnings("unchecked")
    public static <T, U extends T> Class<T> getDomainClass(U domainObject) {
        return domainObject instanceof org.hibernate.proxy.HibernateProxy ? (Class<T>) domainObject
                .getClass().getSuperclass()
                : (Class<T>) domainObject.getClass();
    }

    /**
     * Get the "true" domain class for a given class, i.e. avoid Hibernate proxy
     * classes etc.
     *
     * @param <T>
     *            The domain object super class.
     * @param <U>
     *            The domain object class (possibly hibernate proxy).
     * @param clazz
     *            The domain object class
     * @return The <em>true</em> domain object class.
     */
    @SuppressWarnings("unchecked")
    public static <T, U extends T> Class<T> getDomainClass(Class<U> clazz) {
        Class<?> retClass = org.hibernate.proxy.HibernateProxy.class
                .isAssignableFrom(clazz) ? clazz.getSuperclass() : clazz;
        return (Class<T>) retClass;
    }

    /***
     * Get the domain super class for the given class, or <code>null</code> if
     * no such class exists.
     *
     * @param <T>
     *            The domain object super class
     * @param <U>
     *            The domain object class
     * @param clazz
     *            The domain object class
     * @return The closest super class of the specified domain class or
     *         <code>null</code> if the super class is an <code>Object</code>.
     */
    @SuppressWarnings("unchecked")
    public static <T, U extends T> Class<T> getDomainSuperClass(Class<U> clazz) {
        if (clazz.equals(Object.class)) {
            return null;
        }
        return (Class<T>) clazz.getSuperclass();
    }

    /**
     * Find the lockFlag property methods if implemented for an object instance
     *
     * @param object
     *           The object that should contain the methods
     *
     * @return array containing methods. [0] is the get method, [1] is the set method
     *         Both may be null
     */
    static private Method[] getLockFlagPropertyMethods(Object object) {
        Method getMethod = null;
        Method setMethod = null;
        try {
            getMethod = object.getClass().getMethod("getLockFlag",new Class[] {});
            setMethod = object.getClass().getMethod("setLockFlag",new Class[] {Serializable.class});
        } catch (NoSuchMethodException e) {
            // Ignore, this implies that one or both methods are not implemented
        } catch (SecurityException e) {
            throw new G9BaseException(e);
        }
        return new Method[] {getMethod,setMethod};
    }

    /**
     * Test whether an object has a "lockFlag" field or not. If the
     * g9.config.property file has a property "DomainUtil.ignoreLockFlag"
     * with the value "true" or "1", the return value is always false.
     *
     * @param domainObject
     *            the object to test
     * @return true if the specified object has a lock flag.
     */
    public static boolean hasLockFlag(Object domainObject) {
        if (DomainUtil.ignoreLockFlag) {
            return false;
        }
        if (domainObject instanceof LockFlag) {
            return true;
        }
        Method[] methods = getLockFlagPropertyMethods(domainObject);
        return methods[0] != null && methods[1] != null;
    }

    /**
     * Gets the lock flag property of the domain object. If the domain object
     * does not have a lock flag, <code>null</code> is returned.
     *
     * Gets the lock flag property of the domain object. If the domain object
     * does not have a lock flag, <code>null</code> is returned.
     *
     * @param domainObject
     *            (missing javadoc)
     * @return (missing javadoc)
     */
    public static Serializable getLockFlag(Object domainObject) {
        if (DomainUtil.ignoreLockFlag) {
            return null;
        }
        if (domainObject instanceof LockFlag) {
            return ((LockFlag)domainObject).getLockFlag();
        }
        Method[] methods =  getLockFlagPropertyMethods(domainObject);
        if (methods[0] == null || methods[1] == null) {
            return null;
        }
        try {
            return  (Serializable)methods[0].invoke(domainObject,new Object[] {});
        } catch (IllegalArgumentException e) {
            throw new G9BaseException(e);
        } catch (IllegalAccessException e) {
            throw new G9BaseException(e);
        } catch (SecurityException e) {
            throw new G9BaseException(e);
        } catch (InvocationTargetException e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * Sets the lock flag property on the domain object. If the domain object
     * does not have a lock flag, no action is taken.
     *
     * Sets the lock flag property on the domain object. If the domain object
     * does not have a lock flag, no action is taken.
     *
     * @param domainObject
     *            (missing javadoc)
     * @param lockFlag
     *            (missing javadoc)
     */
    public static void setLockFlag(Object domainObject, Serializable lockFlag) {
        if (DomainUtil.ignoreLockFlag) {
            return;
        }
        if (domainObject instanceof LockFlag) {
            ((LockFlag)domainObject).setLockFlag(lockFlag);
	    return;
        }
        Method[] methods =  getLockFlagPropertyMethods(domainObject);
        if (methods[0] == null || methods[1] == null) {
            return;
	}
	try {
            methods[1].invoke(domainObject,new Object[] {lockFlag});
        } catch (IllegalArgumentException e) {
            throw new G9BaseException(e);
        } catch (IllegalAccessException e) {
            throw new G9BaseException(e);
        } catch (SecurityException e) {
            throw new G9BaseException(e);
        } catch (InvocationTargetException e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * Return true if two lockFlags are equal
     *
     * @param flag1
     *            (missing javadoc)
     * @param flag2
     *            (missing javadoc)
     * @return true if two lockFlags are equal
     */
    public static boolean equalsLockFlag(Serializable flag1, Serializable flag2) {
        if (flag1 == null || flag2 == null) {
            return false;
        }
        if (!flag1.getClass().equals(flag2.getClass())) {
            return false;
        }
        if (flag1.getClass().isArray()) {
            for (int ix = 0; ix < Array.getLength(flag1); ++ix) {
                Object elem1 = Array.get(flag1, ix);
                Object elem2 = Array.get(flag2, ix);
                if (!elem1.equals(elem2)) {
                    return false;
                }
            }

            return true;
        }
        return flag1.equals(flag2);
    }


    /**
     * Checks if the object in question is not initialized.
     *
     * @param o
     *            the object to check
     * @return <code>true</code> if the object is not initialized.
     */
    public static boolean isLazy(Object o) {
        if (o instanceof Set<?>) {
            return isLazy((Set<?>) o);
        }
        if (o == null) {
            return false;
        }
        try {
            o.hashCode();
        } catch (LazyInitializationException e) {
            return true;
        }
        return false;
    }

    /**
     * Checks if the Set in question contains un-initialized lazy objects.
     *
     * @param s
     *            the Set to check
     * @return <code>true</code> if the set contains un-initialized objects.
     */
    public static boolean isLazy(Set<? extends Object> s) {
        if (s == null) {
            return false;
        }
        if (s instanceof PersistentSet) {
            PersistentSet pSet = (PersistentSet) s;
            return !pSet.wasInitialized();
        }
        try {
            Iterator<? extends Object> it = s.iterator();
            if (it.hasNext()) {
                Object o = it.next();
                o.hashCode();
            }
        } catch (LazyInitializationException e) {
            return true;
        }
        return false;
    }

    /**
     * Strips the specified domain object. All fields in the domain object are
     * accessed (using reflection and a bean-style get-method, e.g.
     * <code>getFoo()</code>) and tested for the following conditions:
     *
     * <ol>
     * <li>The field name is not in the set of fields to ignore.
     * <li>The field value is a (direct or indirect) sub-class of
     * {@link PersistentClass}.
     * </ol>
     *
     * If both conditions are met, the field value is set to <code>null</code>
     * (again using reflection and a bean-style set-method, e.g.
     * <code>setFoo(null)</code>.
     *
     * <p>
     * <strong>Note:</strong>All exceptions that are caught as part of the
     * reflective access to the domain object are wrapped in a
     * G9BaseException and re-thrown.
     *
     * @param <T>
     *            The domain object (super) class
     * @param <U>
     *            The domain object type
     *
     * @param domainObject
     *            the domain object to strip
     * @param domainClass
     *            the declared domain object class
     * @param ignore
     *            the set of relation-names to ignore when stripping.
     * @return the striped domain object.
     */
    @SuppressWarnings("unchecked")
    public static <T, U extends T> U strip(U domainObject,
            Class<T> domainClass, Set<String> ignore) {

        if (domainObject != null) {

            Object[] extractedMethods = extractMethods(domainClass, ignore);
            Method[] getMethods = (Method[]) extractedMethods[0];
            Map<String, Method> setMethods = (Map<String, Method>) extractedMethods[1];

            for (int i = 0; i < getMethods.length; i++) {
                Method method = getMethods[i];
                String methodName = method.getName();

                try {
                    Object fieldValue = getProperty(domainObject, method);

                    if (shouldStrip(fieldValue)) {
                        Method setMethod = setMethods
                                .get(asPropertyName(methodName));
                        Object[] args = { null };
                        setMethod.invoke(domainObject, args);
                    }

                } catch (IllegalArgumentException e) {
                    throw new G9BaseException(e);
                } catch (IllegalAccessException e) {
                    throw new G9BaseException(e);
                } catch (SecurityException e) {
                    throw new G9BaseException(e);
                } catch (InvocationTargetException e) {
                    throw new G9BaseException(e);
                }
            }
        }

        return domainObject;

    }

    /**
     * Strips the specified domain object recursively, i.e. strip the domain
     * object and all domain objects which are referenced from this object. All
     * fields in the domain object are accessed (using reflection and a
     * bean-style get-method, e.g. <code>getFoo()</code>) and tested for the
     * following condition:
     *
     * <ol>
     * <li>The field value is a (direct or indirect) sub-class of
     * {@link PersistentClass}.
     * </ol>
     *
     * If the condition is met, the field is accessed by invoking its
     * get-method. If this causes a Hibernate LazyInitializationException,
     * either directly or via an InvocationTargetException, the field value is
     * set to <code>null</code> (again using reflection and a bean-style
     * set-method, e.g. <code>setFoo(null)</code>.
     *
     * <p>
     * <strong>Note:</strong>All exceptions except those mentioned above that
     * are caught as part of the reflective access to the domain object are
     * wrapped in a G9BaseException and re-thrown.
     *
     * @param domainObject
     *            - the object to strip recursively
     * @param parent
     *            - the parent object, avoided when traversing the object graph
     *            recursively
     */
    @SuppressWarnings("unchecked")
    public static void stripRecursive(Object domainObject, Object parent) {
        Set<String> dummyIgnore = Collections.emptySet();
        Class<?> domainClass = DomainUtil.getDomainClass(domainObject);
        Object[] extractedMethods = DomainUtil.extractMethods(domainClass,
                dummyIgnore);
        Method[] getMethods = (Method[]) extractedMethods[0];
        Map<String, Method> setMethods = (Map<String, Method>) extractedMethods[1];

        for (int i = 0; i < getMethods.length; i++) {
            Method method = getMethods[i];
            Method setMethod = null;
            try {
                Object fieldValue = getProperty(domainObject, method);

                if (shouldStrip(fieldValue)) {
                    setMethod = setMethods
                            .get(asPropertyName(method.getName()));
                    Object related = fieldValue;
                    if (related instanceof Collection) {
                        Collection<Object> s = (Collection<Object>) related;
                        Iterator<Object> it = s.iterator();
                        while (it.hasNext()) {
                            related = it.next();
                            stripRecursive(related, domainObject);
                        }
                    } else {
                        if (related != null && related != parent) {
                            related.hashCode(); // Trigger
                            // LazyInitializationException
                            stripRecursive(related, domainObject);
                        }
                    }
                }

            } catch (IllegalArgumentException e) {
                throw new G9BaseException(e);
            } catch (IllegalAccessException e) {
                throw new G9BaseException(e);
            } catch (SecurityException e) {
                throw new G9BaseException(e);
            } catch (InvocationTargetException e) {
                setProperty(domainObject, setMethod, null);
            } catch (LazyInitializationException e) {
                setProperty(domainObject, setMethod, null);
            }
        }
    }

    private static <T> Object[] extractMethods(Class<T> domainClass,
            Set<String> ignore) {

        Method[] methods = domainClass.getMethods();

        // Make sure we don't run into any problems...
        AccessibleObject.setAccessible(methods, true);

        List<Method> tmpGetMethods = new ArrayList<Method>();
        Map<String, Method> tmpSetMethods = new HashMap<String, Method>();
        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            String methodName = method.getName();
            if (ignore(methodName, ignore)) {
                continue;
            }

            if (isSetMethod(methodName)) {
                tmpSetMethods.put(asPropertyName(methodName), method);
            } else if (isGetMethod(methodName)) {
                tmpGetMethods.add(method);
            }
        }
        Object[] retVal = new Object[2];
        retVal[0] = tmpGetMethods.toArray(new Method[] {});
        retVal[1] = tmpSetMethods;
        return retVal;
    }

    private static boolean ignore(String methodName, Set<String> ignore) {
        return methodName.length() > 3
                && ignore.contains(asPropertyName(methodName));
    }

    private static boolean isGetMethod(String methodName) {
        return methodName.startsWith("get");
    }

    private static boolean isSetMethod(String methodName) {
        return methodName.startsWith("set");
    }

    private static boolean shouldStrip(Object fieldValue) {
        return fieldValue != null
                && (isPersistentClass(fieldValue) || isCollection(fieldValue));
    }

    private static boolean isCollection(Object object) {
        return object instanceof Collection<?>;
    }

    private static Object getProperty(Object domainObject, Method getMethod)
            throws SecurityException, IllegalAccessException,
            InvocationTargetException {

        Object[] args = {};
        try {
            return getMethod.invoke(domainObject, args);
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    private static void setProperty(Object domainObject, Method setMethod,
            Object value) {
        if (domainObject != null && setMethod != null) {
            try {
                Object[] args = { value };
                setMethod.invoke(domainObject, args);
            } catch (InvocationTargetException ie) {
                throw new G9BaseException(ie);
            } catch (IllegalArgumentException ie) {
                throw new G9BaseException(ie);
            } catch (IllegalAccessException ie) {
                throw new G9BaseException(ie);
            } catch (SecurityException ie) {
                throw new G9BaseException(ie);
            }
        }
    }

    private static String asPropertyName(String methodName) {
        if (methodName.length() <= 3) {
            return null;
        }
        String propName = String.valueOf(Character.toLowerCase(methodName
                .charAt(3)));
        if (methodName.length() > 4) {
            propName += methodName.substring(4);
        }

        return propName;
    }

    @SuppressWarnings("deprecation")
	private static boolean isPersistentClass(Object object) {
        return object instanceof PersistentClass;
    }
}
