/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.util.LinkedList;
import java.util.List;

import no.g9.message.administration.IMessageAdmin;

/**
 * The default implementation of the <code>IMessageAdmin</code> interface.
 */
public class MessageAdmin implements IMessageAdmin {

    /** The list of all providers. */
    private List<MessageConfigurator> providers = new LinkedList<MessageConfigurator>();

    /** The list of message loggers. */
    private List<MessageLogger> loggers = new LinkedList<MessageLogger>();

    /** The default logger name. */
    private static String defaultLoggerName = "no.g9";

    /** The file name for the default message file. */
    private static String messageFileName = "message/g9.runtime.messages";

    /** The message interactor. */
    private MessageInteractor msgInteractor;

    /**
     * Setup with default providers, loggers and interactor.
     */
    public MessageAdmin() {
        this.providers.add(new PropertyFileConfigurator(MessageAdmin.messageFileName));
        this.loggers.add(new SimpleLogger(MessageAdmin.defaultLoggerName));
        this.msgInteractor = new SimpleMsgInteractor();
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#addLogger(no.g9.message.log.IMsgLogger)
     */
    @Override
    public void addLogger(MessageLogger logger) {
        this.loggers.add(logger);
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#addProvider(no.g9.message.provider.IMsgProvider)
     */
    @Override
    public void addProvider(MessageConfigurator provider) {
        this.providers.add(provider);
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#clearLoggers()
     */
    @Override
    public void clearLoggers() {
        this.loggers.clear();
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#clearProviders()
     */
    @Override
    public void clearProviders() {
        this.providers.clear();
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#getInteractor()
     */
    @Override
    public MessageInteractor getInteractor() {
        return this.msgInteractor;
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#getLoggers()
     */
    @Override
    public List<MessageLogger> getLoggers() {
        return this.loggers;
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#getProviders()
     */
    @Override
    public List<MessageConfigurator> getProviders() {
        return this.providers;
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#setInteractor(no.g9.message.interaction.IMsgInteractor)
     */
    @Override
    public void setInteractor(MessageInteractor msgInteractor) {
        this.msgInteractor = msgInteractor;
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#setLoggers(java.util.List)
     */
    @Override
    public void setLoggers(List<MessageLogger> loggers) {
        if (loggers == null) {
            throw new NullPointerException("loggers can not be null");
        }
        this.loggers = loggers;
    }

    /* (non-Javadoc)
     * @see no.g9.message.administration.IMessageAdmin#setProviders(java.util.List)
     */
    @Override
    public void setProviders(List<MessageConfigurator> providers) {
        if (providers == null) {
            throw new NullPointerException("providers can not be null");
        }
        this.providers = providers;
    }

    /**
     * @return the <code>String</code> used as default Logger name
     */
    public static String getDefaultLoggerName() {
        return MessageAdmin.defaultLoggerName;
    }

    /**
     * @param defaultLoggerName the new default <code>Logger</code> name
     */
    public static void setDefaultLoggerName(final String defaultLoggerName) {
        MessageAdmin.defaultLoggerName = defaultLoggerName;
    }

}
