/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.util.LinkedList;
import java.util.List;

import no.esito.util.BeanID;
import no.esito.util.ServiceLoader;

import org.springframework.beans.factory.annotation.Autowired;


/**
 * This class dispatches messages to the registered loggers.
 */
@BeanID("messageLogDispatcher")
public class MessageLogDispatcher {

	private List<MessageLogger> messageLoggers=new LinkedList<MessageLogger>();

    /**
      * @param messageLoggers the messageLoggers to set
     */
    @Autowired(required = false)
    public void setMessageLoggers(List<MessageLogger> messageLoggers) {
		this.messageLoggers = messageLoggers;
	}

	/**
	 * Add a logger to the list of message loggers.
	 * @param messageLogger The logger to add.
	 */
	public void addMessageLogger(MessageLogger messageLogger){
		this.messageLoggers.add(messageLogger);
	}

    /**
	 * Remove a logger from the list of message loggers.
	 * @param messageLogger The logger to remove.
	 */
	public void removeMessageLogger(MessageLogger messageLogger){
		this.messageLoggers.remove(messageLogger);
	}


	/**
	 * Get the single instance of this class.
	 * @return The instance.
	 */
	public static MessageLogDispatcher getInstance(){
		MessageLogDispatcher retVal = ServiceLoader
        .<MessageLogDispatcher> getService(MessageLogDispatcher.class);
		return retVal;
	}

	/**
	 * Dispatches this message to all the registered loggers.
	 * @param message The message to dispatch.
	 * @param dispatcherContext The context in which to log.
	 */
	public void log(Message message, DispatcherContext dispatcherContext){
		for (MessageLogger messageLogger : messageLoggers) {
			messageLogger.log(message, dispatcherContext);
		}
	}

	/**
	 * @return The message loggers.
	 */
	List<MessageLogger> getMessageLoggers() {
		return messageLoggers;
	}
}