/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import no.esito.log.Logger;

/**
 * Provider for handling messages to users and to the log.
 */
public class PropertyFileConfigurator implements MessageConfigurator {

    private static final Logger log = Logger.getLogger(PropertyFileConfigurator.class);

    /** Map of messages. */
    private final Map<String, MessageText> messages = new HashMap<String, MessageText>();

    /** Map of message types. */
    private final Map<String, String> messageTypes = new HashMap<String, String>();

    /** Map of message reply sets. */
    private final Map<String, String> messageReplySets = new HashMap<String, String>();

    /** Map of message titles. */
    private final Map<String, String> messageTitles = new HashMap<String, String>();

    /** Our properties. */
    private Properties props;

    /**
     * The separator between parts of a message in the message file.
     */
    private String messagePartSep = ";";

    /**
     * The file name for the message file.
     */
    private String messageFileName;

    private boolean isDefaultConfigurator = false;

    /**
     * Constructor
     *
     * @param fileName Name of the message file
     */
    public PropertyFileConfigurator(final String fileName) {
        this(fileName, false);
    }

    /**
     * Construct a new PropertyFileConfigurater. Messages are read from the specified fileName.
     *
     * @param fileName name of property file
     * @param isDefaultConfigurator if <code>true</code>, this configurator will always allow other configurators to continue
     *        configuring a message.
     */
    public PropertyFileConfigurator(final String fileName, boolean isDefaultConfigurator) {
        setMessageFileName(fileName);
        loadMessages(getMessageFileName());
        log.debug("Loading " + fileName);
        this.isDefaultConfigurator = isDefaultConfigurator;
    }

    /**
     * Fill in data for the given <code>Message</code>. If not found in the <code>messages</code> cache it is loaded from the
     * <code>Property</code> instance.
     *
     * @see MessageConfigurator#configure(Message)
     */
    @Override
    public synchronized boolean configure(Message msg) {
        String messageID = msg.getMessageID();
        MessageText msgSummaryText = messages.get(messageID);
        if (msgSummaryText == null) {
            String propVal = props.getProperty(messageID);
            if (propVal != null) {
                String[] propParts = propVal.split(this.messagePartSep, 4);
                msgSummaryText = new MessageText(messageID, propParts[3]);
                messages.put(messageID, msgSummaryText);
                messageTypes.put(messageID, propParts[0]);
                messageReplySets.put(messageID, propParts[1]);
                messageTitles.put(messageID, propParts[2]);
                
            } else {
                return false;
            }
        }
                
        msg.setMessageText(msgSummaryText.substitute(true, msg.getArgs()));
        msg.setMessageID(messageID);
        msg.setLogLevel(CLogLevels.LOG_CENTRAL);

        MessageType msgType = MessageTypeEnum.fromString(messageTypes.get(messageID));
        if (msgType != null) {
            msg.setMsgType(msgType);
        }

        ReplySetType msgReplySet = ReplySetType.fromString(messageReplySets.get(messageID));
        if (msgReplySet != null) {
            msg.setValidReplies(msgReplySet);
        }

        MessageText msgTitleText = new MessageText(messageID, messageTitles.get(messageID));
        msg.setTitle(msgTitleText.substitute(true, msg.getArgs()));
        return !isDefaultConfigurator;
    }

    /**
     * @return the message file name.
     */
    public String getMessageFileName() {
        return this.messageFileName;
    }

    /**
     * Set the name of the message file.
     *
     * @param messageFileName (missing javadoc)
     */
    public void setMessageFileName(final String messageFileName) {
        this.messageFileName = messageFileName;
    }

    /**
     * @return the separator between parts of the message.
     */
    public String getMessagePartSep() {
        return this.messagePartSep;
    }

    /**
     * Set the separator between parts of the message.
     *
     * @param messagePartSep (missing javadoc)
     */
    public void setMessagePartSep(final String messagePartSep) {
        this.messagePartSep = messagePartSep;
    }

    /**
     * Load messages from the given file name to the <code>Property</code> instance.
     *
     * @param fileName the name of the property file.
     */
    private void loadMessages(String fileName) {
        try {
            props = new Properties();
            InputStream in = null;
            in = this.getClass().getClassLoader().getResourceAsStream(fileName);
            if (in == null) {
                String message = "Unable to open message file: " + fileName;
                throw new RuntimeException(message);
            }
            props.load(in);
            in.close();
        } catch (IOException ex) {
            String message = "Error reading from message file: " + fileName;
            throw new RuntimeException(message, ex);
        }
    }

}
