/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.util.HashMap;
import java.util.Map;

/**
 * Enumeration of valid message replies.
 * 
 */
public class ReplySetType extends MessageEnums {
	
    /** Keeps list of created values. */
    private final static Map<Integer, ReplySetType> values = new HashMap<Integer, ReplySetType>();
    
    /** Keeps list of created names. */
    private final static Map<String, ReplySetType> names = new HashMap<String, ReplySetType>();

    /** Keeps list of valid message reply types */
    private MessageReplyType msgReplyTypeSet[]; 
    
    /** Default reply set to use if msgReplyTypeSet is null */
    private static final MessageReplyType defaultMsgReplySet[] = 
        {MessageReplyType.REPLY_OK};

    /**
     * Constructor for internal use only.
     * 
     * @param pCode Numeric value.
     * @param pName Textual value.
     * @param msgReplyTypeSet reply type set
     */
    protected ReplySetType(final int pCode, final String pName,
            final MessageReplyType msgReplyTypeSet[]) {
        super(pCode, pName);
        this.msgReplyTypeSet = (msgReplyTypeSet != null)?
                msgReplyTypeSet:defaultMsgReplySet;        
        values.put(this.code, this);
        names.put(this.name, this);
    }

    /**
     * Returns the constant corresponding to the given integer, or null if none
     * found..
     * 
     * @param pCode Integer value.
     * @return Corresponding constant.
     */
    public static ReplySetType fromInt(final int pCode) {
        return values.get(Integer.valueOf(pCode));
    }

    /**
     * Returns the constant corresponding to the given name, or null if none
     * found..
     * 
     * @param pName String name.
     * @return Corresponding constant.
     */
    public static ReplySetType fromString(final String pName) {
        return names.get(pName);
    }

    /**
     * Returns the message reply type set
     * @return the message reply type set
     */
    public MessageReplyType[] toReplySet() {
        return msgReplyTypeSet.clone();
    }
    
    /** Reply set: None */
    private static final MessageReplyType none[] = {MessageReplyType.REPLY_NONE};   

    /** Reply set: OK */
    private static final MessageReplyType ok[] = {MessageReplyType.REPLY_OK};
    
    /** Reply set: OK, Cancel */
    private static final MessageReplyType okCancel[] = {MessageReplyType.REPLY_OK, 
        MessageReplyType.REPLY_CANCEL};
        
    /** Reply set: Yes, No. */
    private static final MessageReplyType yesNo[] = {MessageReplyType.REPLY_YES, 
        MessageReplyType.REPLY_NO};
    
    /** Reply set: Yes, No, Cancel. */
    private static final MessageReplyType yesNoCancel[] = {MessageReplyType.REPLY_YES, 
        MessageReplyType.REPLY_NO, MessageReplyType.REPLY_CANCEL};
    
    /** Reply set: Abort, Retry, Ignore. */
    private static final MessageReplyType abortRetryIgnore[] = {MessageReplyType.REPLY_ABORT,
        MessageReplyType.REPLY_RETRY, MessageReplyType.REPLY_IGNORE};

    /** Reply set: OK, Info. */
    private static final MessageReplyType okInfo[] = {MessageReplyType.REPLY_OK,
        MessageReplyType.REPLY_INFO};
    
    /** Legal replies: Input an existing file. */
    public static final ReplySetType REPLSET_FILE_EXISTS = new ReplySetType(
            -100, "FileExists", null);

    /** Legal replies: Input an existing directory. */
    public static final ReplySetType REPLSET_DIR_EXISTS = new ReplySetType(
            -101, "DirExists", null);

    /** Legal replies: Input a file name (may not exist). */
    public static final ReplySetType REPLSET_FILE = new ReplySetType(-102,
            "File", null);

    /** Legal replies: Input a directory name (may not exist). */
    public static final ReplySetType REPLSET_DIR = new ReplySetType(-103, "Dir", null);

    /** Legal replies: Input a date. */
    public static final ReplySetType REPLSET_DATE = new ReplySetType(-2, "Date", null);
    
    /** Legal replies: Input a string. */
    public static final ReplySetType REPLSET_STRING = new ReplySetType(-1,
            "String", okCancel);
    
    /** Legal replies: None. */
    public static final ReplySetType REPLSET_NONE = new ReplySetType(0, "None", none);
    
    /** Legal replies: OK. */
    public static final ReplySetType REPLSET_OK = new ReplySetType(1, "OK", ok);

    /** Legal replies: OK, Cancel. */
    public static final ReplySetType REPLSET_OKCANCEL = new ReplySetType(2,
            "OKCancel", okCancel);

    /** Legal replies: Yes, No. */
    public static final ReplySetType REPLSET_YESNO = new ReplySetType(3,
            "YesNo", yesNo);

    /** Legal replies: Yes, No, Cancel. */
    public static final ReplySetType REPLSET_YESNOCANCEL = new ReplySetType(4,
            "YesNoCancel", yesNoCancel);

    /** Legal replies: Abort, Retry, Ignore. */
    public static final ReplySetType REPLSET_ABORTRETRYIGNORE = new ReplySetType(5,
            "AbortRetryIgnore", abortRetryIgnore);

    /** Legal replies: OK, Info. */
    public static final ReplySetType REPLSET_OKINFO = new ReplySetType(6,
            "OKInfo", okInfo);
}
