/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Standard parsing algorithm for message texts.
 * 
 */
public final class StdParser {

    /** Start character of variable */
    static String cStart = "<";

    /** Separator between variable parts. */
    static String cSep = ";";

    /** Terminate character for variable. */
    static String cEnd = ">";

    /** Complete (compiled) pattern for a variable. */
    static Pattern variableRe = Pattern.compile(cStart + "[^" + cEnd + "]+"
            + cEnd);

    /** Complete (compiled) pattern for a linefeed. */
    static String lfRe = "\\\\n";

    /** No objects of this class! */
    private StdParser() {
        /* EMPTY */
    }

    /**
     * Parses the input string.
     * 
     * @param raw Input text
     * @param result A sequence of ITextParts (FixedText, Newline, NumVar,
     *            SpecialVar).
     */
    static public void parse(final String raw, final List<ITextPart> result) {
        if (raw == null) return;
        int lastEnd = 0;
        final Matcher m = variableRe.matcher(raw);
        while (m.find()) {
            final int starts = m.start();
            final int ends = m.end();
            if (starts > lastEnd) {
                parseFixed(raw.substring(lastEnd, starts), result);
            }
            lastEnd = ends;
            result.add(parseVar(raw.substring(starts + 1, ends - 1)));
        }
        if (lastEnd < raw.length()) {
            parseFixed(raw.substring(lastEnd), result);
        }
        if (result.size() == 0) {
            result.add(new FixedTextPart(""));
        }
    }

    /**
     * Parse the contents of a variable, return a corresponding object.
     * 
     * @param varTxt Variable contents.
     * @return NumVar or SpecialVar representing the contents.
     */
    private static ITextPart parseVar(final String varTxt) {
        final String[] parts = new String[] { "", "", "", "" };
        final String[] parts1 = varTxt.split(cSep, parts.length);
        for (int i = 0; i < parts1.length; i++) {
            parts[i] = parts1[i];
        }
        if (parts[0].matches("[0-9]+")) {
            return new NumVarPart(Integer.parseInt(parts[0]), parts[1],
                    parts[2], parts[3]);
        }
        return new SpecialVarPart(varTxt);
    }

    /**
     * Parse a fixed string part.
     * 
     * @param fix The fixed text input
     * @param result See {@link #parse(String, List)}
     */
    private static void parseFixed(final String fix, final List<ITextPart> result) {
        final String[] parts = fix.split(lfRe, -1);
        for (int i = 0; i < parts.length; i++) {
            if (i >= 1) {
                result.add(new NewLinePart());
            }
            if (!"".equals(parts[i])) {
                result.add(new FixedTextPart(parts[i]));
            }
        }
    }

}
