/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message.metric;

import java.io.Serializable;

/**
 * This class contains simple metrics, as the name strongly suggests...
 * Each metric is identified by a <code>String</code> key, and contains a number
 * representing "number of" and a number representing "value of".
 * Additionally, a timer is "started" (recorded) when the object is created.
 *
 */
public class SimpleMetric implements IMetric, Serializable {
 /*
 * I considered splitting this into two separate classes - timers and counters -
 * but I find that when I time something, I generally need to count it as well,
 * and often the need to time something worth counting comes up.  Thus...
 */

    /** A counter of all created instances. */
    private static int instances = 0;

    /** A key to retrieve this value by. */
    private String key;

    /** The startup time (in milliseconds since Epoch). */
    protected long started = System.currentTimeMillis();

    /**
     * Timer end (in milliseconds since Epoch).
     * @see #setEnd()
     */
    protected long ended;

    /** Logs how many times the value is read (intermediate time #). */
    protected long lNo;

    /** Number of... */
    private long no = 0;

    /** Value/sum. */
    private long val = 0;

    /** Default constructor. */
    protected SimpleMetric() {
        this.key= "SimpleMetric#" + (instances);
        incrementInstances();
    }

    /**
     * Constructor providing ID only.
     * @param pkey ID
     */
    public SimpleMetric(final String pkey)  {
        incrementInstances();
        this.key= pkey;
    }

    /**
     * Constructor setting all values.
     * @param s    ID
     * @param n    Value of  number of times called (normally 0)
     * @param v    Value so far (normally 0)
     */
    public SimpleMetric(final String s, final long n, final long v)  {
        this(s);
        this.no=  n;
        this.val= v;
    }
    
    private static synchronized void incrementInstances() {
        ++instances;
    }

    /**
     * Captures the current time in {@link #ended} for internal operations.
     */
    protected void setEnd()  {
        this.ended= System.currentTimeMillis();
        this.lNo++;
    }

    /*
     * Makes a clone of this object.
     *
     * @return a deep copy of this object
     * @throws CloneNotSupportedException Shouldn't happen!
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    /**
     * Add a value to a metric and increase the counter.
     * @param v    Value to add
     */
    @Override
    public void add(final long v)  {
        this.no++;
        this.val+=v;
    }

    /**
     * Adds the current elapsed time in milliseconds as if with {@link #add},
     * and restarts the timer.
     */
    public void addMS()  {
        this.no++;
        this.val += getTimer();
        reset();
    }

    /** Resets the metric. */
    @Override
    public void clear()  {
        this.started= System.currentTimeMillis();
        this.lNo= 0;
        this.no=  0;
        this.val= 0;
    }

    /** Resets the start time for the timer functions. */
    public void reset()  {
        this.started= System.currentTimeMillis();
    }

    /**
     * Returns the metric key value.
     *
     * @return key value
     * @see #setKey
     */
    @Override
    public String getKey()  {
        return this.key;
    }

    /**
     * Gets the count of updates to this metric.
     *
     * @return count of updates
     * @see #setCount
     */
    @Override
    public long getCount()  {
        return this.no;
    }

    /**
     * Gets the total value for this metric.
     *
     * @return total value
     * @see #setValue
     */
    @Override
    public long getValue()  {
        return this.val;
    }

    /**
     * Returns the running time so far.
     *
     * @return running time in millis
     */
    @Override
    public long getTimer()  {
        setEnd(); // Set ended and lNo
        return this.ended-this.started;
    }

    /**
     * Sets the end value for the timer and returns a reference to this object.
     *
     * @return this object
     */
    @Override
    public IMetric stopTimer()  {
        setEnd();
        return this;
    }

    /**
     * Sets the metric key value.
     * <b>Note</b>:  Should not be called after saving in e.g. a HashMap,
     * since this changes the hash value of the object.
     * @param newVal    New key value.  Must not be null.
     * @throws IllegalArgumentException If key is null.
     * @see #getKey
     *
     */
    @Override
    public void setKey(final String newVal)  {
        if ( newVal==null ) {
            throw new IllegalArgumentException("null key");
        }
        this.key= newVal;
    }

    /**
     * Sets the counter value.
     * @param n    New counter value
     * @see #getCount
     *
     */
    @Override
    public void setCount(final long n)  {
        this.no= n;
    }

    /**
     * Sets the new value without changing the count.
     * @param n New value
     * @see #getValue
     *
     */
    @Override
    public void setValue(final long n)  {
        this.val= n;
    }

    /**
     * Calculates average - value per number.
     *
     * @return average value, or 0.0 if neither fields are set,
     *    or <code>Double.MAX_VALUE</code> if only number is unset.
     */
    public double avg()  {
        if ( this.no==0 ) {
            if ( this.val==0 ) {
                return 0.0d;
            }
            return Double.MAX_VALUE;
        }
        return ((double)this.val)/((double)this.no);
    }

    /**
     * Calculates speed - number per second.
     *
     * @return speed in number per second, or 0.0 if the timer has not been
     * stopped, or <code>Double.MAX_VALUE</code> if started==ended.
     */
    public double speed()  {
        if ( this.ended==0 ) {
            return 0.0d;
        }
        if ( this.started==this.ended ) {
            return Double.MAX_VALUE;
        }
        return (1000.0d * this.no) /
               (1.0d * (this.ended-this.started));
    }

    /**
     * Calculates volume - value per second.
     *
     * @return volume in number per second, or 0.0 if the timer has not been
     * stopped, or <code>Double.MAX_VALUE</code> if started==ended.
     */
    public double volume()  {
        if ( this.ended==0 ) {
            return 0.0d;
        }
        if ( this.started==this.ended ) {
            return Double.MAX_VALUE;
        }
        return (1000.0d * this.val) /
               (1.0d * (this.ended-this.started));
    }

    @Override
    public String toString()  {
        final StringBuffer sb= new StringBuffer(this.key);

        setEnd(); // Set ended and lNo

        sb.append("\t");
        sb.append(this.lNo);
        sb.append("\t");
        sb.append((this.started-this.ended));
        sb.append("\t");
        sb.append(this.no);
        sb.append("\t");
        sb.append(this.val);
        return sb.toString();
    }

    /**
     * Returns a hash code for this object.
     *
     * @return hash code based on key
     */
    @Override
    public int hashCode()  {
        return this.key.hashCode();
    }

    /*
     * Compares this object to another of the same class.
     *
     * @param that The object to compare to
     * @return TRUE if this object is equal to the other object
     *
     */
    @Override
    public boolean equals(final Object that)  {
        if (that==null) return false;
        if (that==this) return true;
        if (this.getClass() != that.getClass()) return false;
        return this.key.equals(((SimpleMetric)that).key);
    }
}
