/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service;

import java.io.Serializable;

import no.g9.support.ClientContext;
import no.g9.support.FindData;
import no.g9.support.ObjectSelection;

/**
 * Wrapper for parameters to JGrape service method calls. Used when serializing
 * the parameters to a log file, and later when the log is read and the service
 * calls repeated.
 */
public class JGrapeParamWrapper implements Serializable {

	private String action;

	private ClientContext ctx;

	private ObjectSelection objectSelection;

	private FindData findData;

	private String dbMapping;

	private long when;

	/**
	 * Create a new wrapper for the given action, setting the system time in
	 * the <code>when</code> field.
	 * 
	 * @param action - the action for this wrapper.
	 */
	public JGrapeParamWrapper(String action) {
		this.action = action;
		this.when = System.currentTimeMillis();
	}

	/**
	 * Create a new wrapper for the given action, client context and object
	 * selection.
	 * 
	 * @param action - the action for this wrapper.
	 * @param ctx - the client context.
	 * @param objectSelection - the object selection.
	 */
	public JGrapeParamWrapper(String action, ClientContext ctx, ObjectSelection objectSelection) {
		this(action);
		this.ctx = ctx;
		this.objectSelection = objectSelection;
	}
    	
    /**
     * Create a new wrapper for the given action, client context, find data
     * and database mapping.
     * 
     * @param action - the action for this wrapper.
     * @param ctx - the client context.
     * @param findData - the find data settings.
     * @param dbMapping - the database mapping name.
     */
	public JGrapeParamWrapper(String action, ClientContext ctx, FindData findData, String dbMapping) {
		this(action);
		this.ctx = ctx;
		this.findData = findData;
		this.dbMapping = dbMapping;
	}

	/**
	 * @return the action.
	 */
	public String getAction() {
		return action;
	}

	/**
	 * @return the client context.
	 */
	public ClientContext getCtx() {
		return ctx;
	}

	/**
	 * @return the object selection.
	 */
	public ObjectSelection getObjectSelection() {
		return objectSelection;
	}

	/**
	 * @return the find data settings.
	 */
	public FindData getFindData() {
		return findData;
	}

	/**
	 * @return the database mapping name.
	 */
	public String getDbMapping() {
		return dbMapping;
	}

	/**
	 * @return the system time for when the wrapper was created.
	 */
	public long getWhen() {
		return when;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
    public String toString() {
        String actionName = action == null ? "no action" : action;
        String roleName = objectSelection == null ? "no role" : objectSelection
                .getTargetRoleName();
        String osName = objectSelection == null ? "no os" : objectSelection
                .getObjectSelectionName();
        return "JGrapeService[" + actionName + ", " + roleName + ", " + osName
                + "], time:" + when;
	}

}
