/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * Client context for server requests.
 */
public class ClientContext implements Serializable {

    /** Current user. */
    private String userId;

    /**
     * The logical name of the database context, used in
     * <code>no.g9.dataaccess.SessionFactory</code> and configuration files.
     */
    private String databaseContext;

    /**
     * Debug mode. Might be loglevel, but most often default or debug, thus
     * debug.
     */
    private boolean debug = false;

    /** Additional parameters */
    private Map<Object, Object> params = new HashMap<Object, Object>();

    /** Current application name */
    private String applicationName;

    /** Default constructor. */
    public ClientContext() {
        /* EMPTY */
    }

    /**
     * Copy constructor.
     *
     * @param ctx the <code>ClientContext</code> to copy
     * @param deepCopy true if a deep copy is wanted, i.e. create a copy of the
     *            <code>params</code> Map
     */
    public ClientContext(ClientContext ctx, boolean deepCopy) {
        this.applicationName = ctx.applicationName;
        this.databaseContext = ctx.databaseContext;
        this.debug = ctx.debug;
        if (deepCopy) {
            this.params = new HashMap<Object, Object>(ctx.params);
        } else {
            this.params = ctx.params;
        }
        this.userId = ctx.userId;
    }

    /**
     * Access method for the userId property.
     *
     * @return the current value of the userId property
     */
    public String getUserId() {
        return this.userId;
    }

    /**
     * Sets the value of the userId property.
     *
     * @param aUserId the new value of the userId property
     */
    public void setUserId(String aUserId) {
        this.userId = aUserId;
    }

    /**
     * Determines if the debug property is true.
     *
     * @return <code>true</code> if the debug property is true
     */
    public boolean getDebug() {
        return this.debug;
    }

    /**
     * Sets the value of the debug property.
     *
     * @param aDebug the new value of the debug property
     */
    public void setDebug(boolean aDebug) {
        this.debug = aDebug;
    }

    /**
     * Set new value of a parameter.
     *
     * @param key Parameter ID.
     * @param value Parameter value.
     */
    public void setParam(final Object key, final Object value) {
        this.params.put(key, value);
    }

    /**
     * Returns a given parameter (or <code>null</code> if not found).
     *
     * @param key Parameter ID.
     * @return Parameter value.
     */
    public Object getParam(final Object key) {
        return this.params.get(key);
    }

    /**
     * @return the current <code>databaseContext</code>
     */
    public String getDatabaseContext() {
        return this.databaseContext;
    }

    /**
     * Set the <code>databaseContext</code> to be used.
     *
     * @param databaseContext (missing javadoc)
     */
    public void setDatabaseContext(final String databaseContext) {
        this.databaseContext = databaseContext;
    }

    /**
     * Set the application name to be used.
     *
     * @param applicationName Application name
     */
    public void setApplicationName(final String applicationName) {
        this.applicationName = applicationName;
    }

    /**
     * @return the current application name
     */
    public String getApplicationName() {
        return this.applicationName;
    }

}
