/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * Class used for formatting dates. Intenrnal use.
 */
public class CustomDate {

    static private String dateInputFormat = "dd.MM.yyyy";

    static private String timeInputFormat = "HH:mm";

    static private String timestampInputFormat = dateInputFormat + " " + timeInputFormat;

    static private String dateOutputFormat = "dd.MM.yyyy";

    static private String timeOutputFormat = "HH:mm";

    static private String timestampOutputFormat = dateOutputFormat + " " + timeOutputFormat;

    static private String dateInternalFormat = "dd.MM.yyyy";

    static private String timeInternalFormat = "HH:mm:ss.SSS";

    static private String timestampInternalFormat = dateInternalFormat + " " + timeInternalFormat;

    static private DateFormat internalDateFormatter;

    static private DateFormat internalTimeFormatter;

    static private DateFormat internalTimestampFormatter;

    /**
     * @param datatype (missing javadoc)
     * @return the default datetime input format according to datatype
     */
    public static String getDatetimeInputFormat(int datatype) {
        String format = null;
        switch (datatype) {
        case G9Consts.DT_DATE:
            format = dateInputFormat;
            break;
        case G9Consts.DT_TIME:
            format = timeInputFormat;
            break;
        case G9Consts.DT_TIMESTAMP:
            format = timestampInputFormat;
            break;
        default:
            break;
        }
        return format;
    }

    /**
     * Set the default datetime input format according to datatype
     *
     * @param datatype (missing javadoc)
     * @param format (missing javadoc)
     */
    public static void setDatetimeInputFormat(int datatype, String format) {
        switch (datatype) {
        case G9Consts.DT_DATE:
            dateInputFormat = format;
            break;
        case G9Consts.DT_TIME:
            timeInputFormat = format;
            break;
        case G9Consts.DT_TIMESTAMP:
            timestampInputFormat = format;
            break;
        default:
            break;
        }
    }

    /**
     * @param datatype (missing javadoc)
     * @return the default datetime output format according to datatype
     */
    public static String getDatetimeOutputFormat(int datatype) {
        String format = null;
        switch (datatype) {
        case G9Consts.DT_DATE:
            format = dateOutputFormat;
            break;
        case G9Consts.DT_TIME:
            format = timeOutputFormat;
            break;
        case G9Consts.DT_TIMESTAMP:
            format = timestampOutputFormat;
            break;
        default:
            break;
        }
        return format;
    }

    /**
     * Set the default datetime output format according to datatype
     *
     * @param datatype (missing javadoc)
     * @param format (missing javadoc)
     */
    public static void setDatetimeOutputFormat(int datatype, String format) {
        switch (datatype) {
        case G9Consts.DT_DATE:
            dateOutputFormat = format;
            break;
        case G9Consts.DT_TIME:
            timeOutputFormat = format;
            break;
        case G9Consts.DT_TIMESTAMP:
            timestampOutputFormat = format;
            break;
        default:
            break;
        }
    }

    /**
     * @param datatype (missing javadoc)
     * @return the default datetime internal format according to datatype
     */
    public static String getDatetimeInternalFormat(int datatype) {
        String format = null;
        switch (datatype) {
        case G9Consts.DT_DATE:
            format = dateInternalFormat;
            break;
        case G9Consts.DT_TIME:
            format = timeInternalFormat;
            break;
        case G9Consts.DT_TIMESTAMP:
            format = timestampInternalFormat;
            break;
        default:
            break;
        }
        return format;
    }

    /**
     * @param datatype (missing javadoc)
     * @return the internal datetime formatter according to datatype
     */
    public static DateFormat getDatetimeInternalFormatter(int datatype) {
        DateFormat formatter = null;
        switch (datatype) {
        case G9Consts.DT_DATE:
            formatter = internalDateFormatter;
            if (formatter == null) {
                formatter = new SimpleDateFormat(getDatetimeInternalFormat(datatype));
                formatter.setLenient(false);
                internalDateFormatter = formatter;
            }
            break;
        case G9Consts.DT_TIME:
            formatter = internalTimeFormatter;
            if (formatter == null) {
                formatter = new SimpleDateFormat(getDatetimeInternalFormat(datatype));
                formatter.setLenient(false);
                internalTimeFormatter = formatter;
            }
            break;
        case G9Consts.DT_TIMESTAMP:
            formatter = internalTimestampFormatter;
            if (formatter == null) {
                formatter = new SimpleDateFormat(getDatetimeInternalFormat(datatype));
                formatter.setLenient(false);
                internalTimestampFormatter = formatter;
            }
            break;
        default:
            break;
        }
        return formatter;
    }

    /**
     * Format a date according to the internal format.
     *
     * @param datatype (missing javadoc)
     * @param date (missing javadoc)
     * @return (missing javadoc)
     */
    public static synchronized String formatInternal(int datatype, Object date) {
        return getDatetimeInternalFormatter(datatype).format(date);
    }

    /**
     * Parses a date string in internal format according to datatype.
     *
     * @param datatype (missing javadoc)
     * @param str (missing javadoc)
     * @return (missing javadoc)
     * @throws ParseException (missing javadoc)
     */
    public static synchronized Date parseInternal(int datatype, String str) throws ParseException {
        return getDatetimeInternalFormatter(datatype).parse(str);
    }

    /**
     * Performs completion of dates on lostfocus on date fields Needs to be
     * changed according to dateformat and custom rules
     *
     * @param d the string representation of the date
     * @return the formatted date.
     */
    public static String superDateFormat(String d) {
        if (d == null) {
            return null;
        }
        int length = d.length();
        if (length == 10 || length == 0) {
            return d;
        }
        GregorianCalendar gNow = new GregorianCalendar();
        Date now = new Date();
        gNow.setTime(now);
        int firstDot = d.indexOf('.');
        int lastDot = d.indexOf('.', firstDot + 1);
        if (length <= 3) {
            int day = 0;
            if (firstDot == -1) {
                day = Integer.parseInt(d);
            } else {
                day = Integer.parseInt(d.substring(0, firstDot));
            }
            /*
             * Set day always - if day is before - then assume next month
             */
            if (day < gNow.get(Calendar.DAY_OF_MONTH)) {
                gNow.add(Calendar.MONTH, 1);
            }
            gNow.set(Calendar.DAY_OF_MONTH, day);
            return getDateFormatter().format(gNow.getTime()); // '
        }
        if (length <= 6) {
            int day = Integer.parseInt(d.substring(0, firstDot));
            int month = 0;
            if (lastDot == -1 || firstDot == lastDot) {
                month = Integer.parseInt(d.substring(firstDot + 1, length));
            } else {
                month = Integer.parseInt(d.substring(firstDot + 1, lastDot));
            }
            if (month < gNow.get(Calendar.MONTH)) {
                gNow.add(Calendar.YEAR, 1);
            }
            gNow.set(Calendar.MONTH, month - 1);
            gNow.set(Calendar.DAY_OF_MONTH, day);
            return getDateFormatter().format(gNow.getTime()); // '
        }
        if (length == 7) {
            int currentYear = gNow.get(Calendar.YEAR);
            String currentYearString = Integer.toString(currentYear);
            if (currentYearString.charAt(3) == d.charAt(length - 1)) {
                StringBuffer buff = new StringBuffer(d);
                buff.insert(buff.length() - 1, currentYearString.charAt(2));
                d = buff.toString();
            }
            SimpleDateFormat formatter = new SimpleDateFormat("dd.MM.yy");
            Date desired = null;
            try {
                desired = formatter.parse(d);
            } catch (ParseException e) {
                String msgID = CRuntimeMsg.CF_DATE_PARSE_MSG;
                Object[] msgArgs = {"no.g9.support.CustomDate.superDateFormat", d, e.getMessage()};
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(e, msg);

            }
            return getDateFormatter().format(desired);
        }
        if (length == 8) {
            SimpleDateFormat formatter = new SimpleDateFormat("dd.MM.yy");
            Date desired = null;
            try {
                desired = formatter.parse(d);
            } catch (ParseException e) {
                String msgID = CRuntimeMsg.CF_DATE_PARSE_MSG;
                Object[] msgArgs = {"no.g9.support.CustomDate.superDateFormat", d, e.getMessage()};
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(e, msg);
            }
            return getDateFormatter().format(desired);
        }
        return d;
    }

    static private SimpleDateFormat dateFormatter = new SimpleDateFormat(getDateFormat());

    /**
     * Returns the date input format string
     *
     * @return String representation of date format
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public static String getDateFormat() {
        return dateInputFormat;
    }

    /**
     * @param datatype (missing javadoc)
     * @return the default datetime input format according to datatype
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public static String getDatetimeFormat(int datatype) {
        return getDatetimeInputFormat(datatype);
    }

    /**
     * Returns the date formatter
     *
     * @return the date formatter
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public static SimpleDateFormat getDateFormatter() {
        return dateFormatter;
    }

    /**
     * Parses a date. Need not be changed for other date formats Returns null if
     * the date is invalid.
     *
     * @param date String representation of the date to parse
     * @return (missing javadoc)
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public static Date parse(String date) {
        try {
            getDateFormatter().setLenient(false);
            return getDateFormatter().parse(date);
        } catch (Exception e) {
            return null;
        }
    }
}
