/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 * Helper class for display rules and formats
 */
public class FormatHelper {

    private static char decimalSeparator = 0;

    private static char groupingSeparator = 0;

    private static Locale locale;

    /** Constant for the floating minus sign */
    public static final int SIGN_FLOATING_MINUS = -2;

    /** Constant for the minus sign */
    public static final int SIGN_MINUS = -1;

    /** Constant for no sign */
    public static final int SIGN_NONE = 0;

    /** Constant for plus sign */
    public static final int SIGN_PLUS = 1;

    /** Constant for floating plus sign */
    public static final int SIGN_FLOATING_PLUS = 2;

    /**
     * Private constructor, only static methods
     */
    private FormatHelper() {
        // Empty.
    }

    /**
     * Add a character to a numeric format string
     *
     * @param format the string buffer
     * @param c the character to add
     * @param signSet turn sign set on or off
     * @return <code>true</code> if the sign is set.
     */
    private static boolean addChar(StringBuffer format, char c, boolean signSet) {
        switch (c) {
        case '*':
        case '9':
            format.append('0');
            break;
        case '+':
        case '-':
            if (!signSet) {
                format.append('+');
                signSet = true;
            } else {
                format.append('#');
            }
            break;
        case 'Z':
            format.append('#');
            break;
        case '.':
        case ',':
            format.append(c);
            break;
        case 'B':
            format.append(' ');
            break;
        default:
            format.append(c);
            break;
        }
        return signSet;
    }

     /**
     * Convert numeric display rule to an internal format
     *
     * @param displayRule (missing javadoc)
     * @return Internal format string
     */
    public static String getNumericFormat(String displayRule) {
        StringBuffer format = new StringBuffer();
        int i;
        char a = 0;
        char b;
        boolean signSet = false;
        for (i = 0; i < displayRule.length(); i++) {
            b = displayRule.charAt(i);
            switch (b) {
            case '9':
            case 'Z':
            case '*':
            case '-':
            case '+':
            case 'B':
            case '/':
            case '.':
            case ',':
                if (a != 0) {
                    signSet = addChar(format, a, signSet);
                }
                a = b;
                break;
            case '(':
                if (a != 0) {
                    int no = 0;
                    b = 0;
                    i++;
                    while (i < displayRule.length()) {
                        b = displayRule.charAt(i);
                        if (!Character.isDigit(b)) {
                            break;
                        }
                        no = no * 10 + Character.digit(b, 10);
                        i++;
                    }
                    if (b == ')') {
                        while (no > 0) {
                            signSet = addChar(format, a,signSet);
                            no--;
                        }
                    }
                    a = 0;
                }
                break;
            default:
                break;
            }
        }
        if (a != 0) {
            addChar(format, a, signSet);
        }
        return format.toString();
    }
    /**
     * Get sign code from a display rule<br>
     * Five possible return values:<br>
     * SIGN_MINUS,SIGN_FLOARING_MINUS, SIGN_PLUS, SIGN_FLOATING_PLUS, SIGN_NONE
     *
     * @param displayRule (missing javadoc)
     * @return sign
     */
    public static int getSign(String displayRule) {
        int sign = SIGN_NONE;
        if (displayRule != null) {
            int index = displayRule.indexOf('-');
            if (index >= 0) {
                sign = displayRule.indexOf('-',index+1) == -1 ? SIGN_MINUS : SIGN_FLOATING_MINUS;
            } else {
                index = displayRule.indexOf('+');
                if (index >= 0) {
                    sign = displayRule.indexOf('+',index+1) == -1 ? SIGN_PLUS : SIGN_FLOATING_PLUS;
                }
            }
        }
        return sign;
    }

    /**
     * Get numeric input format
     *
     * @param displayFormat (missing javadoc)
     * @return Java format string
     */
    public static String getNumericInputFormat(String displayFormat) {
        StringBuffer format = new StringBuffer();
        int i;
        char a;
        boolean decimal = false;
        for (i = 0; i < displayFormat.length(); i++) {
            a = displayFormat.charAt(i);
            switch (a) {
            case '+':
                format.insert(0,'+');
                break;
            case '.':
                decimal = true;
                format.append(a);
                break;
            case '0':
                format.append(a);
                break;
            case '#':
                format.append(decimal ? '0' : a);
                break;
            default:
                break;
            }
        }
        return format.toString();
    }

    /**
     * Convert datetime display rule to Java format. If the given display rule
     * is null or "NLS", null is returned (default format for locale will be
     * used). If the given display rule is of "Systemator numeric" type, an
     * effort is made to find an appropriate Java format.
     *
     * @param displayRule (missing javadoc)
     * @return Java format string
     */
    public static String getDatetimeFormat(String displayRule) {
        String format = displayRule;
        if (format != null) {
            if (format.matches(".*NLS.*")) {
                format = null;
            }
            else if (format.matches(".*[*Z9]+.*")) {
                format = null;
            } else {
                format = format.replaceAll("cc?", ""); // Century, not supported
                format = format.replaceAll("q", ""); // Quarter of year, not supported
                if (format.indexOf("AM") != -1) {
                    format = format.replaceAll("H", "h"); // Hour of day, 1-12
                }
                format = format.replaceAll("AM", "aa"); // AM/PM
                format = format.replaceAll("AD|BC", "G"); // Era
                format = format.replaceAll("wwww", "EEEE"); // Day of week
                format = format.replaceAll("www", "EEE"); // --- " ---
                format = format.replaceAll("w", "F"); // --- " ---
                format = format.replaceAll("m", "�"); // Switch m and M
                format = format.replaceAll("M", "m"); // --- " ---
                format = format.replaceAll("�", "M"); // --- " ---
                format = format.replaceAll("j", "D"); // Day of year
                format = format.replaceAll("W", "w"); // Week of year
                format = format.replaceAll("S\\.SSS", "S.���");// Switch milliseconds
                format = format.replaceAll("S\\.SS", "S.��"); // --- " ---
                format = format.replaceAll("S\\.S", "S.�"); // --- " ---
                format = format.replaceAll("S", "s"); // Seconds
                format = format.replaceAll("�", "S"); // Switch back milliseconds
            }
        }
        return format;
    }

    /**
     * Return a new display rule suitable for the given data type if the given
     * display rule is null or empty.
     *
     * @param datatype (missing javadoc)
     * @param displayrule (missing javadoc)
     * @param displaylength (missing javadoc)
     * @param precision (missing javadoc)
     * @param scale (missing javadoc)
     * @return String (missing javadoc)
     */
    public static String getDisplayrule(int datatype, String displayrule,
            int displaylength, int precision, int scale) {
        if (displayrule == null || displayrule.length() == 0) {
            StringBuffer rule = new StringBuffer();
            switch (datatype) {
            case G9Consts.DT_SHORTINT:
            case G9Consts.DT_LONGINT:
            case G9Consts.DT_LONGLONG:
            case G9Consts.DT_NUMERIC:
            case G9Consts.DT_REAL:
            case G9Consts.DT_DOUBLE:
                if (precision == 0)
                    precision = displaylength;
                else if (displaylength > precision)
                    rule.append('-');
                if (scale > 0) {
                    precision = precision - scale - 1;
                }
                while (precision > 1) {
                    rule.append('-');
                    precision--;
                }
                if (precision == 1) {
                    rule.append('9');
                }
                if (scale > 0) {
                    rule.append('.');
                    while (scale > 0) {
                        rule.append('9');
                        scale--;
                    }
                }
                displayrule = rule.length() > 0 ? rule.toString() : null;
                break;
            case G9Consts.DT_DATE:
            case G9Consts.DT_TIME:
            case G9Consts.DT_TIMESTAMP:
            default:
                displayrule = null;
                break;
            }
        }
        return displayrule;
    }

    /**
     * @return the desimalSeparator
     */
    public static char getDecimalSeparator() {
        char ds = FormatHelper.decimalSeparator;
        if (ds == 0) {
            ds = new DecimalFormatSymbols(FormatHelper.getLocale())
                    .getDecimalSeparator();
        }
        return ds;
    }

    /**
     * @param decimalSeparator the decimalSeparator to set
     */
    public static void setDecimalSeparator(char decimalSeparator) {
        FormatHelper.decimalSeparator = decimalSeparator;
    }

    /**
     * @return the groupingSeparator
     */
    public static char getGroupingSeparator() {
        char gs = FormatHelper.groupingSeparator;
        if (gs == 0) {
            gs = new DecimalFormatSymbols(FormatHelper.getLocale())
                    .getGroupingSeparator();
        }
        return gs;
    }

    /**
     * @param groupingSeparator the groupSeparator to set
     */
    public static void setGroupingSeparator(char groupingSeparator) {
        FormatHelper.groupingSeparator = groupingSeparator;
    }

    /**
     * Return the <code>locale</code> object. If none is set,
     * <code>Locale.getDefault()</code> will be used.
     *
     * @return the locale
     */
    public static Locale getLocale() {
        if (FormatHelper.locale == null) {
            setLocale(Locale.getDefault());
        }
        return FormatHelper.locale;
    }

    /**
     * @param locale the locale to set
     */
    public static void setLocale(Locale locale) {
        FormatHelper.locale = locale;
    }

}
