/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.Serializable;
import java.lang.reflect.Field;

import no.g9.message.*;

/**
 * The super class of all enumerators
 */
public class G9Enumerator implements Serializable, Comparable<G9Enumerator> {

    /** The current enumerator value */
    public int currentValue;

    /**
     * Gets the current enumerator value
     *
     * @return the current enumerator value
     */
    public int getCurrentValue() {
        return currentValue;
    }

    /**
     * Gets the current value as an Integer
     *
     * @return Current Value as Integer.
     */
    public Integer getCurrentValueAsInteger() {
        return Integer.valueOf(getCurrentValue());
    }

    /**
     * Sets the current enumeration value
     *
     * @param n the current enumeration value
     *
     */
    public void setCurrentValue(int n) {
        currentValue = n;
    }

    /**
     * Internal use!
     *
     * @deprecated This method is likely to fail...
     * @return the name of the current value.
     */
    @Deprecated
    public String currentAsName() {
        Field fields[] = getClass().getDeclaredFields();
        Exception ex = null;
        for (int i = 0; i < fields.length; i++) {
            // Work-around for bug #4533479 in java
            fields[i].setAccessible(true);
            try {
                if (!fields[i].getName().equals("currentValue") && fields[i].getType() == int.class && fields[i].getInt(this) == currentValue) {
                    return fields[i].getName(); // <- normal return
                }
            } catch (IllegalArgumentException e) {
                ex = e;
            } catch (IllegalAccessException e) {
                ex = e;
            }
            if (ex != null) {
                // We are here only if an exception was caught
                Object[] msgArgs = {
                        G9Enumerator.class.getClass(),
                        fields[i], this.getClass(), ex.getMessage()
                };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                        msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                ex = null;
            }
        }
        return null;
    }

    /**
     * Assign the current value based on the supplied parameter
     *
     * @param n the name of the enumerator value
     */
    public void assignFromName(String n) {
        Field fields[] = getClass().getDeclaredFields();
        Exception ex = null;
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getName().equals(n)) {
                try {
                    currentValue = fields[i].getInt(this);
                } catch (IllegalArgumentException e) {
                    ex = e;
                } catch (IllegalAccessException e) {
                    ex = e;
                }
                if (ex != null) {
                    Object[] msgArgs = {
                            G9Enumerator.class.getClass(),
                            fields[i], this.getClass(), ex.getMessage()
                    };
                    Message msg = MessageSystem.getMessageFactory().getMessage(
                            CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                            msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    ex = null;
                }
                return;
            }
        }
    }

    /**
     * Gets the key from the specified name
     *
     * @param f the name
     *
     * @return the key, or -1 if the name does not exist
     */
    public int keyFromName(String f) {
        Exception ex = null;
        try {
            return getClass().getField(f).getInt(this);
        } catch (SecurityException e) {
            ex = e;
        } catch (NoSuchFieldException e) {
            ex = e;
        } catch (IllegalArgumentException e) {
            ex = e;
        } catch (IllegalAccessException e) {
            ex = e;
        }
        // We ate here only if an exception was caught
        Object[] msgArgs = {
                G9Enumerator.class.getClass(), f,
                this.getClass(), ex.getMessage()
        };
        Message msg = MessageSystem.getMessageFactory().getMessage(
                CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                msgArgs);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        return -1;
    }

    /**
     * Gets the name from a key
     *
     * @param n the key
     *
     * @return the name, or null if no such key
     * @deprecated This method may return incorrect value.
     */
    @Deprecated
    public String nameFromKey(int n) {
        Field fields[] = getClass().getDeclaredFields();
        Exception ex = null;
        for (int i = 0; i < fields.length; i++) {
            try {
                if (fields[i].getInt(this) == n &&
                        !fields[i].getName().equals("currentValue")) {
                    return fields[i].getName();
                }
            } catch (IllegalArgumentException e) {
                ex = e;
            } catch (IllegalAccessException e) {
                ex = e;
            }
            if (ex != null) {
                Object[] msgArgs = { G9Enumerator.class.getClass(),
                        fields[i], this.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                        msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                ex = null;
            }
        }
        return null;
    }

    @Override
    public int compareTo(G9Enumerator genum) {
        return getCurrentValue() - genum.getCurrentValue();
    }

    @Override
    public boolean equals(Object o) {
        if (o == null) return false;
        if (!(o.getClass().equals(getClass()))) return false;
        return getCurrentValue() == ((G9Enumerator) o).getCurrentValue();
    }

    @Override
    public int hashCode() {
        return getClass().getName().hashCode() + getCurrentValue();
    }
}
