/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.math.BigDecimal;

/**
 * A wrapper for a BigDecimal value. Ensures that the correct scale is preserved
 * for the BigDecimal value. The default scale for all new instances of Numeric
 * is 0 unless given in the constructor. Instances of this class are not
 * immutable.
 */
public class Numeric extends Number implements Cloneable, Comparable<Numeric> {

    private static int defaultScale = 0;

    private BigDecimal value;

    /**
     * Create a new <code>Numeric</code> with zero value and default scale.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public Numeric() {
        this.value = BigDecimal.valueOf(0, Numeric.defaultScale);
    }

    /**
     * Create a new <code>Numeric</code> with given value and default scale.
     * 
     * @param n The <code>Numeric</code> value to copy.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public Numeric(Numeric n) {
        this.value = n.getValue().setScale(Numeric.defaultScale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and default scale.
     * 
     * @param l The <code>long</code> value to set.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public Numeric(long l) {
        this.value = new BigDecimal(String.valueOf(l)).setScale(
                Numeric.defaultScale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and default scale.
     * 
     * @param s A String containing the value to set.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public Numeric(String s) {
        this.value = new BigDecimal(s).setScale(Numeric.defaultScale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and default scale.
     * 
     * @param d A <code>BigDecimal</code> with the value to set.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public Numeric(BigDecimal d) {
        this.value = d.setScale(Numeric.defaultScale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and scale.
     * 
     * @param n The <code>Numeric</code> value to copy.
     * @param scale The scale.
     */
    public Numeric(Numeric n, int scale) {
        this.value = n.getValue().setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and scale.
     * 
     * @param l The <code>long</code> value to set.
     * @param scale The scale.
     */
    public Numeric(long l, int scale) {
        this.value = new BigDecimal(String.valueOf(l)).setScale(scale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and scale.
     * 
     * @param s A String containing the value to set.
     * @param scale The scale.
     */
    public Numeric(String s, int scale) {
        this.value = new BigDecimal(s).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Create a new <code>Numeric</code> with given value and scale.
     * 
     * @param d A <code>BigDecimal</code> with the value to set.
     * @param scale The scale.
     */
    public Numeric(BigDecimal d, int scale) {
        this.value = d.setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Get scale for this Numeric
     * 
     * @return the Numeric's scale
      */
    public int getScale() {
        return value.scale();
    }

    /**
     * Add the given <code>Numeric</code> value to this.
     * 
     * @param n <code>Numeric</code> operand.
     */
    public void add(Numeric n) {
        int scale = value.scale();
        value = value.add(n.getValue()).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Add the given <code>long</code> value to this.
     * 
     * @param l <code>long</code> operand.
     */
    public void add(long l) {
        this.add(new Numeric(l));
    }

    /**
     * Subtract the given <code>Numeric</code> value from this.
     * 
     * @param n <code>Numeric</code> operand.
     */
    public void subtract(Numeric n) {
        int scale = value.scale();
        value = value.subtract(n.getValue()).setScale(scale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Subtract the given <code>long</code> value from this.
     * 
     * @param l <code>long</code> operand.
     */
    public void subtract(long l) {
        this.subtract(new Numeric(l));
    }

    /**
     * Multiply the given <code>Numeric</code> value with this.
     * 
     * @param n <code>Numeric</code> operand.
     */
    public void multiply(Numeric n) {
        int scale = value.scale();
        value = value.multiply(n.getValue()).setScale(scale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Multiply the given <code>long</code> value with this.
     * 
     * @param l <code>long</code> operand.
     */
    public void multiply(long l) {
        this.multiply(new Numeric(l));
    }

    /**
     * Multiplies this numeric with the percent given in <code>n</code>: &lt;this
     * numeric&gt; * n / 100. The division part of this calculation is done with
     * precision of 2.
     * 
     * @param i Percent.
     */
    public void multiplyPercent(int i) {
        int scale = value.scale();
        BigDecimal tmp = BigDecimal.valueOf(i).movePointLeft(2);
        value = value.multiply(tmp).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Divide this with the given <code>Numeric</code> value.
     * 
     * @param n <code>Numeric</code> operand.
     */
    public void divide(Numeric n) {
        value = value.divide(n.getValue(), value.scale(),
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Divide this with the given <code>long</code> value.
     * 
     * @param l <code>long</code> operand.
     */
    public void divide(long l) {
        this.divide(new Numeric(l));
    }

    /**
     * Rounds the value with BigDecimal.ROUND_HALF_UP, precision of 0.
     */
    public void round() {
        int scale = value.scale();
        value = value.setScale(0, BigDecimal.ROUND_HALF_UP).setScale(scale);
    }

    /**
     * Rounds the value by five
     */
    public void roundFives() {
        int scale = value.scale();
        value = value
                .divide(BigDecimal.valueOf(5), 0, BigDecimal.ROUND_HALF_UP);
        value = value.multiply(BigDecimal.valueOf(5)).setScale(scale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Rounds the value by ten
     */
    public void roundTens() {
        int scale = value.scale();
        value = value.divide(BigDecimal.valueOf(10), 0,
                BigDecimal.ROUND_HALF_UP);
        value = value.multiply(BigDecimal.valueOf(10)).setScale(scale,
                BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Return the <code>BigDecimal</code> value as an long.
     * 
     * @return value as long.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public long getLongValue() {
        return value.longValue();
    }

    /**
     * Return the <code>BigDecimal</code> value as a int.
     * 
     * @return value as int.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public int getIntValue() {
        return value.intValue();
    }

    /**
     * Return the <code>BigDecimal</code> as an int where its value is
     * multiplied with 100.
     * 
     * @return 1.50 is returned as 150.
     */
    public int getAsInt100() {
        return value.movePointRight(2).intValue();
    }

    /**
     * Return the <code>BigDecimal</code> value as a double.
     * 
     * @return value as double.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public double getDoubleValue() {
        return value.doubleValue();
    }

    @Override
    public String toString() {
        return value.toString();
    }

    /**
     * Return the unscaled value as a <code>String</code>.
     * 
     * @return unscaled value as <code>String</code>.
     */
    public String toStringWithoutDecimalPoint() {
        return value.unscaledValue().toString();
    }

    /**
     * Return a copy of this numeric, with the same scale.
     * 
     * @return The copied Numeric.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    public Numeric makeCopy() {
        return new Numeric(this, value.scale());
    }

    /**
     * Set a new <code>BigDecimal</code> as the value. The previous scale is
     * preserved.
     * 
     * @param value The new value.
     */
    public void setValue(BigDecimal value) {
        int scale = this.value.scale();
        this.value = value.setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * Return the <code>BigDecimal</code> value.
     * 
     * @return value.
     */
    public BigDecimal getValue() {
        return value;
    }

    /**
     * Compare this to the given <code>long</code> value, using
     * <code>BigDecimal.compareTo()</code>.
     * 
     * @param l value to compare.
     * @return result of comparison.
     */
    public int compareTo(long l) {
        return value.compareTo(BigDecimal.valueOf(l));
    }

    /**
     * Compare this to the given <code>Object</code>, using
     * <code>BigDecimal.compareTo()</code>.
     * 
     * @param ob object to compare.
     * @return result of comparison, 0 if ob is not a <code>Numeric</code>.
     */
    @Override
    public int compareTo(Numeric ob) {
        return value.compareTo(ob.getValue());
    }

    /**
     * Compare this to the given <code>Numeric</code> value, using
     * <code>BigDecimal.equals()</code>.
     * 
     * @param n value to compare.
     * @return result of comparison.
     */
    public boolean equals(Numeric n) {
    	if (n != null) {
    		return n.getValue().equals(this.value);
    	}
    	return false;
    }

    /**
     * Compare this to the given <code>BigDecimal</code> value, using
     * <code>BigDecimal.equals()</code>.
     * 
     * @param d value to compare.
     * @return result of comparison.
     */
    public boolean equals(BigDecimal d) {
        return value.equals(d);
    }

    /**
     * Compare this to the given <code>Object</code>, using
     * <code>BigDecimal.equals()</code>.
     * 
     * @param that object to compare.
     * @return result of comparison, <code>false</code> if <code>ob</code> is
     *         not a <code>Numeric</code>.
     */
    @Override
    public boolean equals(Object that) {
        if (that == null) return false;
        if (this == that) return true;
        if (!getClass().equals(that.getClass())) return false;
        return equals((Numeric) that);
    }
    
    @Override
    public int hashCode() {
        return value.hashCode();
    }

    /**
     * @return the <code>defaultScale</code> property.
     */
    public static int getDefaultScale() {
        return defaultScale;
    }

    /**
     * Sets the default scale
     * 
     * @param scale the new default scale
     */
    public static void setDefaultScale(int scale) {
        defaultScale = scale;
    }

    /**
     * Negate this value
     */
    public void negate() {
        value = value.negate();
    }

    /*
     * @return byte value of this numeric
     * @see java.lang.Number#byteValue()
     */
    @Override
    public byte byteValue() {
        return value.byteValue();
    }

    /*
     * @return double value of this numeric
     * @see java.lang.Number#doubleValue()
     */
    @Override
    public double doubleValue() {
        return value.doubleValue();
    }

    /*
     * @return float value of this numeric
     * @see java.lang.Number#floatValue()
     */
    @Override
    public float floatValue() {
        return value.floatValue();
    }

    /*
     * @return int value of this numeric
     * @see java.lang.Number#intValue()
     */
    @Override
    public int intValue() {
        return value.intValue();
    }

    /*
     * @return long value of this numeric
     * @see java.lang.Number#longValue()
     */
    @Override
    public long longValue() {
        return value.longValue();
    }

    /*
     * @return short value of this numeric
     * @see java.lang.Number#shortValue()
     */
    @Override
    public short shortValue() {
        return value.shortValue();
    }
    
    /**
     * Creates a copy of this object. Overrides Object.clone.
     * 
     * @return A clone of this Object.
     * @see java.lang.Object#clone()
     */
    @Override
    public Object clone() {
        try {
            return super.clone();
        } catch (CloneNotSupportedException e) {
            throw new AssertionError("Clone failed"); // Should not happen
        }
    }

}
