/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;

import no.esito.log.Logger;
import no.g9.exception.G9ServiceException;
import no.g9.service.G9Spring;

/**
 * Retrieves a resource reference from g9 properties.
 */
public class Pathfinder {

    /** Logger. */
    private static Logger log = Logger.getLogger(Pathfinder.class);

    /** Only static methods. */
    private Pathfinder() {
        /* Empty */
    }

    /**
     * Returns a {@link java.net.URL} based on the <code>property</code> and the
     * <code>file</code>. The resource is located as follows: If
     * <code>file</code> is
     * <ul>
     * <li>an absolute path =&gt; that path is used</li>
     * <li>a relative path =&gt; that path is prefixed with the value of
     * <code>property</code>, which is read from Registry. If the value of
     * <code>property</code> is
     * <ul>
     * <li>an absolute path, a URL containing <code>property</code> +
     * <code>file</code> is returned.</li>
     * <li>a relative path, a URL is returned from the class loader.</li>
     * </ul>
     * </li>
     * </ul>
     *
     * If none of the attempts above succeed the event is logged and null is
     * returned
     *
     * The definition of absolute pathname is system dependent. On UNIX systems,
     * a pathname is absolute if its prefix is "/". On Microsoft Windows
     * systems, a pathname is absolute if its prefix is a drive specifier
     * followed by "\\", or if its prefix is "\\".
     * {@link java.io.File#isAbsolute() Ref}
     *
     *
     * @param property The property to retrieve
     * @param file A relative or absolute path to a resource
     * @return URL to the resource
     */
    public static URL getResourceReference(String property, String file) {
        if (log.isDebugEnabled()) {
            log.debug("Getting resource reference for property = " + property
                    + " and file = " + file + ".");
        }
        try {
            File resourceFile = new File(file);

            // Try path: original_path
            if (log.isTraceEnabled()) {
                log.trace("Attempting to find resource using absolute path "
                        + file);
            }
            if (resourceFile.isAbsolute()) {
                log.debug("Resource found using file name as absolute path.");
                return resourceFile.toURI().toURL();
            }

            // Try path: propertyPath/original_path
            File prefix = new File(Pathfinder.getPropertyValue(property));
            resourceFile = new File(prefix, resourceFile.getPath());

            if (log.isTraceEnabled()) {
                log.trace("Attempting to find resource using " + resourceFile);
            }

            if (prefix.isAbsolute()) {
                log.debug("Resource found using priopertyPath/original_path");
                return resourceFile.toURI().toURL();
            }

            URL url = getResourceUsingClassLoader(resourceFile);

            if (url != null) {
                return url;
            }

            if (!isOptional(file)) {
                log.warn("Failed to get URL for resource " + file + ".");
            }

            return null;

        } catch (MalformedURLException e) {
            log.error("Malformed URL.", e);
            return null;
        } catch (NullPointerException e) {
            log.error("A nullpointer exception was encountered.", e);
            return null;
        }
    }

    private static URL getResourceUsingClassLoader(File resourceFile) {
        ClassLoader cl = ClassLoader.getSystemClassLoader();
        String path = resourceFile.getPath();

        if (path.startsWith(File.separator)) {
            path = path.substring(1);
        }

        // Replace backslash with forward slash.
        path = path.replace(File.separatorChar, '/');
        if (log.isTraceEnabled()) {
            log.trace("Attempting to find resource using system class loader "
                    + "classpath/propertyPath/original_path: " + path);
        }

        URL url = getURLFromClassLoader(cl, path);
        if (url != null) {
            return url;
        }

        cl = Thread.currentThread().getContextClassLoader();
        if (log.isTraceEnabled()) {
            log.trace("Attempting to find resource using context class loader "
                    + "classpath/propertyPath/original_path: " + path);
        }
        url = getURLFromClassLoader(cl, path);

        return url;

    }

    private static URL getURLFromClassLoader(ClassLoader classLoader, String path) {
        if (classLoader.getResource(path) != null) {
            if (log.isDebugEnabled()) {
                log.debug("Resource found using " + path);
            }
            return classLoader.getResource(path);
        }
        return null;
    }

    /**
     * Takes a URL, and returns the absolute filename (and path) of that file
     *
     * @param url The URL to get the path for
     * @return The absolute path of the file as String
     */
    public static String getAbsoluteFilePath(URL url) {
        try {
            File tmpFile = new File(url.toURI());
            return tmpFile.getAbsolutePath();
        } catch (IllegalArgumentException e) {
            // File might be located in a jar.
            throw new G9ServiceException(e);
        } catch (URISyntaxException e) {
            log.error("Malformed URI syntax.", e);
            throw new G9ServiceException(e);
        } catch (Exception e) {
            // Just in case the impossible happens
            log.error("Unexpected Exception.", e);
            throw new G9ServiceException(e);
        }
    }

    /**
     * Returns the value of the property.
     *
     * @param property the property name.
     * @return value of the property or empty string if the property isn't found
     */
    private static String getPropertyValue(String property) {
        if (property == null) {
            return "";
        }

        String regProp= Registry.getRegistry().getG9Property(property);
        return (regProp != null) ? regProp : "";
    }

    /**
     * Test if the specified resource is an optional resource.
     * @param resource the name of the resource.
     * @return <code>true</code> if the resource is optional.
     */
    private static boolean isOptional(String resource) {
        for (String definition : G9Spring.getDefaultBeanDefinitions()) {
            if (resource.equals(definition)) {
                return true;
            }
        }

        for (String definition : G9Spring.getOptionalBeanDefinitions()) {
            if (resource.equals(definition)) {
                return true;
            }
        }
        return false;

    }

}
