/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.util.HashMap;
import java.util.Map;

import no.g9.exception.G9BaseException;
import no.g9.message.*;

/**
 * The "one stop" registry for an application. Maintains several
 * <code>RegistrySet</code> properties, either read from the main property
 * file (<code>PROPERTIES_FILE</code>) or from their own files.
 *
 */
public class Registry extends RegistrySet {

    private static Registry regInstance;

    private static final String PROPERTIES_FILE = "g9.config";

    /** (missing javadoc) */
    public static final String SRV_APP_PRE = "ServiceApplication";

    /** (missing javadoc) */
    public static final String DEFAULT_SRV = "DefaultServicePackage";

    /** (missing javadoc) */
    public static final String DEFAULT_DBM = "DefaultDatabaseMapping";

    /** (missing javadoc) */
    public static final String DEFAULT_DBC = "DefaultDatabaseContext";

    /** (missing javadoc) */
    public static final String DEFAULT_XMLCONV_PACKAGE = "DefaultXmlConvertPackage";

    /** (missing javadoc) */
    public static final String DEFAULT_SPREADSHEET_PACKAGE = "DefaultSpreadsheetExporterPackage";

    private static final String DBC_PRE = "Database.Context";

    private static final String DBS_PRE = "Database.Server";

    private static final String DBM_PRE = "Database.Mapping";

    private Map<String, Map<String, RegistrySet>> propertyTypes =
        new HashMap<String, Map<String, RegistrySet>>();

    private Registry() {
        super(PROPERTIES_FILE, null);
    }

    /**
     * Get an application property. There is one <code>RegistrySet</code> for
     * each application (using its own property file).
     *
     * @param fileName the name of the property file name (without extention).
     * @param propName the name of the property to get.
     * @return the found property value.
     */
    public String getApplicationProperty(final String fileName,
            final String propName) {
        return getRegistrySet(SRV_APP_PRE, fileName).getProperty(propName);
    }

    /**
     * Get a database context property. The property is found in the main
     * property file, prefixed with <code>DBC_PRE</code>.
     *
     * @param context the name of the database context.
     * @param propName the name of the property to get.
     * @return the found property value.
     */
    public String getDatabaseContextProperty(final String context, final String propName) {
        return getG9Property(DBC_PRE + "." + context + "." + propName);
    }

    /**
     * Get a database server property. The property is found in the main
     * property file, prefixed with <code>DBS_PRE</code>.
     *
     * @param server the name of the database server.
     * @param propName the name of the property to get.
     * @return the found property value.
     */
    public String getDatabaseServerProperty(final String server, final String propName) {
        return getG9Property(DBS_PRE + "." + server + "." + propName);
    }

    /**
     * Get a database mapping property. The property is found in the main
     * property file, prefixed with <code>DBM_PRE</code>.
     *
     * @param mapping the name of the database mapping.
     * @param propName the name of the property to get.
     * @return the found property value.
     */
    public String getDatabaseMappingProperty(final String mapping, final String propName) {
        return getG9Property(DBM_PRE + "." + mapping + "." + propName);
    }

    /**
     * Get a property from the main property file.
     *
     * @param propName the name of the property to get.
     * @return the found property value.
     */
    public String getG9Property(final String propName) {
        String propValue = props.getProperty(propName);
        return propValue;
    }

    /**
     * Check if a property is set in the main property file
     *
     * @param propName the name of the propery
     * @return {@code true} if the property is set, otherwise {@code false}
     */
    public boolean hasG9Propery(String propName) {
        return propName != null && props.getProperty(propName) != null;
    }

    /**
     * Get a property from the given <code>RegistrySet</code> type.
     *
     * @param setType the type of the registry set.
     * @param setName the name of the registry set.
     * @param propName the name of the property.
     * @return the found property.
     */
    public String getProperty(final String setType, final String setName,
            final String propName) {
        return getRegistrySet(setType, setName).getProperty(propName);
    }

    /**
     * Get the <code>RegistrySet</code> for the given type and name.
     *
     * @param setType the type of the registry set.
     * @param setName the name of the registry set.
     * @return the found <code>RegistrySet</code>.
     */
    public RegistrySet getRegistrySet(final String setType, final String setName) {
        Map<String, RegistrySet> typeMap = propertyTypes.get(setType);
        if (typeMap == null) {
            typeMap = new HashMap<String, RegistrySet>();
            propertyTypes.put(setType, typeMap);
        }
        RegistrySet registrySet = typeMap.get(setName);
        if (registrySet == null) {
            registrySet = new RegistrySet(setName, null);
            typeMap.put(setName, registrySet);
        }
        return registrySet;
    }

    /**
     * @return (missing javadoc)
     */
    public static Registry getRegistry() {
        if (regInstance == null) {
            try {
                regInstance = new Registry();
            } catch (G9BaseException e) {
            	MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(e.getErrMsg());
                throw e;
            }
        }
        return regInstance;
    }

    /**
     * This is a singleton, clone is not supported.
     *
     * @return (missing javadoc)
     * @throws CloneNotSupportedException (missing javadoc)
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

}
