/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.Properties;

import no.g9.exception.G9BaseException;
import no.g9.message.*;

/**
 * Holds one set of property values from a property file.
 *
 */
public class RegistrySet {

    /**
     * The property set as read from a file.
     */
    protected Properties props;

    private static final String PROPERTIES_EXTENSION = ".properties";

    /**
     * Creates a new RegistrySet for the input name
     *
     * @param name the name of the property file (less extension).
     * @param path the path to where the file is to be found.
     */
    public RegistrySet(String name, String path) {
        String fullName = name + PROPERTIES_EXTENSION;
        if (path == null || path.equals("")) {
            props = RegistrySet.loadProperties(fullName, true);
        } else {
            props = RegistrySet.loadProperties(path + fullName, false);
        }
    }

    /**
     * Get a property value for the given key.
     *
     * @param propName the name (key) of the property.
     * @return String
     */
    public String getProperty(String propName) {
        String propValue = props.getProperty(propName);
        return propValue;
    }

    /**
     * Load properties from the given file name.
     *
     * @param fileName the name of the property file.
     * @param useClassLoader true if the <code>classLoader</code> is to be used.
     * @return properties loaded from the file.
     */
    public static Properties loadProperties(final String fileName,
            final boolean useClassLoader) {
        Properties props = new Properties();
        try {
            InputStream in = null;
            if (useClassLoader) {
                in = RegistrySet.class.getClassLoader()
                        .getResourceAsStream(fileName);
            } else {
                in = RegistrySet.class.getResourceAsStream(fileName);
            }

            if (in != null) {
                props.load(in);
                in.close();
            }
        } catch (IOException ex) {
            Object[] args = { fileName };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.IO_FILE_READ_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9BaseException(msg);
        }
        return props;
    }

    /**
     * Returns the iterator for the property entries.
     * @return the name and value {@link Entry} for the properties.
     */
    public Iterator<Entry<Object, Object>> iterator() {
        return props.entrySet().iterator();
    }
}
