/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.File;
import java.math.BigDecimal;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import no.esito.log.Logger;
import no.esito.util.NumberFormat;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.service.G9Spring;
import no.g9.support.convert.AttributeConverter;
import no.g9.support.convert.ConvertException;
import no.g9.support.convert.JasperConvertContext;

import org.joda.time.DateMidnight;
import org.joda.time.DateTime;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

/**
 * Home for some misc. utility functions used for report generation.
 */

@SuppressWarnings("deprecation")
public class ReportHelper {
    
    /** Constant for the image pre-path property */
    private static final String IMAGE_PRE_PATH = "ImagePrePath";

    /** Constant name for the unchecked check button image property */
    public static final String PROPERTY_FILE_NAME_CHECKBOX_UNCHECKED =
            "CheckButton.Image.Checked.FileName";

    /** Constant name for the checked check button image property */
    public static final String PROPERTY_FILE_NAME_CHECKBOX_CHECKED =
            "CheckButton.Image.Unchecked.FileName";

    private final static String STRING_PRESENTATION_OF_NULL = "";

    private static Logger log = Logger.getLogger(ReportHelper.class);

    private static String CHECKBOX_TRUE_DEFAULT_IMAGE_FILENAME =
            "checkbox_true.jpg";
    private static String CHECKBOX_FALSE_DEFAULT_IMAGE_FILENAME =
            "checkbox_false.jpg";
    
    private static final String JODA_DATETIME= "org.joda.time.DateTime";
    private static final String JODA_DATEMIDNIGHT= "org.joda.time.DateMidnight";
    private static final String JAVATIME_LOCALDATE= "java.time.LocalDate";
    private static final String JAVATIME_LOCALTIME= "java.time.LocalTime";
    private static final String JAVATIME_LOCALDATETIME= "java.time.LocalDateTime";

    /*
     * This class is not instanciable
     */
    private ReportHelper() {
        // Not instanciable
    }
    
    /**
     * Formats an Object as a String according to a display rule.
     * 
     * @deprecated Use {@link #format(Object, String, int, String, int, int, int, int, boolean, String)} instead.
     * @param data The data to format
     * @param displayRule Displayrule to apply to the result
     * @param genovaDataType Type of the data. See
     *            no.g9.support.GenovaConsts
     * @param precision Desires precision
     * @param scale Scale
     * @param caseConversion Case conversion
     * @param fieldLength (missing javadoc)
     * @param blankWhenZero Indicates if the field should be blank when value is
     *            zero
     * @return Formatted version of the data
     */
    @Deprecated
    public static String format(Object data, String displayRule, int genovaDataType, int precision,
            int scale, int caseConversion, int fieldLength, boolean blankWhenZero) {
        return format(data, displayRule, genovaDataType, null, precision, scale, caseConversion,
                fieldLength, blankWhenZero, null);
    }

    /**
     * Formats an Object as a String according to a display rule.
     *
     * @param data The data to format
     * @param displayRule Displayrule to apply to the result
     * @param genovaDataType Type of the data. See
     *            no.g9.support.GenovaConsts
     * @param modelType mapped modeltype
     * @param precision Desires precision
     * @param scale Scale
     * @param caseConversion Case conversion
     * @param fieldLength (missing javadoc)
     * @param blankWhenZero Indicates if the field should be blank when value is
     *            zero
     * @param converterBeanName AttributeConverter bean name
     * @return Formatted version of the data
     */
    public static String format(Object data, String displayRule,
            int genovaDataType, String modelType, int precision, int scale, int caseConversion,
            int fieldLength, boolean blankWhenZero, String converterBeanName) {
        if (log.isTraceEnabled()) {
            Object[] args =
                    { "data", data, "displayRule", displayRule,
                            "genovaDataType", genovaDataType, "precision",
                            precision, "scale", scale, "caseConversion",
                            caseConversion, "fieldLength", fieldLength,
                            "blankWhenZero", blankWhenZero };
            String parameters = Logger.formatParameters(args);
            log.trace("formatting: " + parameters);
        }
        if (data == null)
            return STRING_PRESENTATION_OF_NULL;
        
        switch (genovaDataType) {
        case G9Consts.DT_SHORTINT:
        case G9Consts.DT_LONGINT:
        case G9Consts.DT_LONGLONG:
            // Make it a Long...
            if (data instanceof String) {
                try {
                    data = Integer.decode((String) data);
                } catch (NumberFormatException e) {
                    log.debug("Assuming un-parsable shortint/longint (\""
                            + data + "\") is pre-formatted.");
                    return (String) data;
                }
            } else if (!(data instanceof Integer)) {
                log.error("Can not format " + data.getClass()
                        + " to Long (String or Integer expected)");
                return data.toString();
            }
            break;
        case G9Consts.DT_NUMERIC:
        case G9Consts.DT_REAL:
        case G9Consts.DT_DOUBLE:
            // Make it a BigDecimal...
            if (data instanceof String) {
                try {
                    data = new BigDecimal((String) data);
                } catch (NumberFormatException e) {
                    log.debug("Assuming un-parsable numeric/real/double (\""
                            + data + "\") is pre-formatted.");
                    return (String) data;
                }
            } else if (!(data instanceof Long)) {
                log.error("Can not format " + data.getClass()
                        + " to BigDecimal (String or BigDecimal expected)");
                return data.toString();
            }
            break;
        case G9Consts.DT_TEXT:
        case G9Consts.DT_VARTEXT:
            // Make it a String...
            if (!(data instanceof String))
                data = data.toString();
            break;
        case G9Consts.DT_BOOLEAN:
            // Make it a Boolean...
            if (data instanceof String) {
                if (data.equals("true"))
                    data = Boolean.TRUE;
                else if (data.equals("false"))
                    data = Boolean.FALSE;
                else {
                    log.debug("Assuming unknown value for boolean (\"" + data
                            + "\") is pre-formatted.");
                    return (String) data;
                }
            } else if (!(data instanceof Boolean)) {
                log.error("Can not format " + data.getClass()
                        + " to Boolean (String or Boolean expected)");
                return data.toString();
            }
            break;
        case G9Consts.DT_ENUMERATION:
            // Make it a String, and set genova type to text...
            if (!(data instanceof String))
                data = data.toString();
            genovaDataType = G9Consts.DT_TEXT;
            break;
        case G9Consts.DT_DATE:
        case G9Consts.DT_TIME:
        case G9Consts.DT_TIMESTAMP:
            // Make it a Date...
            if (data instanceof String) {
            	String sdata= (String)data;
                try {
                	switch (modelType) {
					case JAVATIME_LOCALDATE:
						data= LocalDate.parse(sdata);
						break;
					case JAVATIME_LOCALTIME:
						data= LocalTime.parse(sdata);
						break;
					case JAVATIME_LOCALDATETIME:
						data= LocalDateTime.parse(sdata);
						break;
					case JODA_DATETIME:
						data= DateTime.parse(sdata);
						break;
					case JODA_DATEMIDNIGHT:
						data= DateMidnight.parse(sdata);
						break;
					default:
						SimpleDateFormat df= new SimpleDateFormat();
						data= df.parse(sdata);
					}
                } catch (IllegalArgumentException | DateTimeParseException | ParseException e) {
                    log.info("Assuming un-parsable date/time/timestamp (\""
                            + data + "\") is pre-formatted.");
                    return (String) data;
                }
            } else if (!(data instanceof Date)) {
                log.error("Can not format " + data.getClass()
                        + " to Date (String or Date expected)");
                return data.toString();
            }
            break;
        case G9Consts.DT_BLOB:
            // We dont support blobs in reports....
            log.info("Unsupported data type in report (blob)");
            return STRING_PRESENTATION_OF_NULL;
        default:
            log.error("Unknown data type: " + genovaDataType);
            return STRING_PRESENTATION_OF_NULL;
        }
        
        if (converterBeanName != null) {
            AttributeConverter<Object, Object> converter= G9Spring.getBean(converterBeanName);
            JasperConvertContext context= new JasperConvertContext(displayRule, genovaDataType,
                    precision, scale, caseConversion, fieldLength, blankWhenZero);
            
            Object fromModel= null;
            try {
                log.debug("Calling converter "+converterBeanName);
                fromModel= converter.fromModel(data, context);
            } catch (ConvertException e) {
                e.printStackTrace();
            }
            
            if (!converter.getTargetType().isAssignableFrom(converter.getModelType()) ||
                    converter.getTargetType().equals(String.class)) {
                return (fromModel != null) ? fromModel.toString() : null;
            }
            if (fromModel == null) return null;
            data= fromModel;
        }
        
        // Transform display rule to java format...
        displayRule =
                FormatHelper.getDisplayrule(genovaDataType, displayRule,
                        fieldLength, precision, scale);
        switch (genovaDataType) {
        case G9Consts.DT_SHORTINT:
        case G9Consts.DT_LONGINT:
        case G9Consts.DT_LONGLONG:
        case G9Consts.DT_NUMERIC:
        case G9Consts.DT_REAL:
        case G9Consts.DT_DOUBLE:
            NumberFormat numberFormat =
                    new NumberFormat(genovaDataType, displayRule);
            return numberFormat.format(
                    FormatHelper.getNumericFormat(displayRule), data, false);
        case G9Consts.DT_DATE:
        case G9Consts.DT_TIME:
        case G9Consts.DT_TIMESTAMP:
            String javaFormat = FormatHelper.getDatetimeFormat(displayRule);
            Locale locale = Locale.getDefault();
            DateFormat df;
            if (javaFormat == null) {
                if (genovaDataType == G9Consts.DT_DATE) {
                    df= DateFormat.getDateInstance(DateFormat.MEDIUM);
                } else if (genovaDataType == G9Consts.DT_TIME) {
                    df= DateFormat.getTimeInstance(DateFormat.MEDIUM);
                } else {
                    df= DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);
                }
            } else {
                df = new SimpleDateFormat(javaFormat, locale);
            }

            if (JODA_DATETIME.equals(modelType)) {
                return df.format(((DateTime)data).toDate());
            }
            if (JODA_DATEMIDNIGHT.equals(modelType)) {
                return df.format(((DateMidnight)data).toDate());
            }
            if (JAVATIME_LOCALDATE.equals(modelType)) {
            	LocalDate ld= (LocalDate)data;
            	return (javaFormat != null) ? ld.format(DateTimeFormatter.ofPattern(javaFormat)) : ld.toString();
			}
            if (JAVATIME_LOCALTIME.equals(modelType)) {
            	LocalTime lt= (LocalTime)data;
            	return (javaFormat != null) ? lt.format(DateTimeFormatter.ofPattern(javaFormat)) : lt.toString();
            }
            if (JAVATIME_LOCALDATETIME.equals(modelType)) {
            	LocalDateTime ldt= (LocalDateTime)data;
            	return (javaFormat != null) ? ldt.format(DateTimeFormatter.ofPattern(javaFormat)) : ldt.toString();
            }
            return df.format(data);

        case G9Consts.DT_TEXT:
        case G9Consts.DT_VARTEXT:
        case G9Consts.DT_LONGTEXT:

            return formatToString((String) data, fieldLength, caseConversion);
        case G9Consts.DT_ENUMERATION:
            return data.toString();
        }

        throw new G9BaseException("Failed to format " + data + " to string");
    }

    private static String formatToString(String data, int fieldLength,
            int caseConversion) {
        if (data == null) {
            return "";
        }
        String result = data.trim();
        if (fieldLength < result.length()) {
            result = result.substring(0, fieldLength);
        }

        switch (caseConversion) {
        case -1:
            result = result.toLowerCase(Locale.getDefault());
            break;
        case 1:
            result = result.toUpperCase(Locale.getDefault());
            break;
        default:
            break;
        }
        return result;
    }

    /**
     * Lookup an enum's label using the ordinal string
     *  
     * @param data - containing enum ordinal
     * @param labels - comma delimited list of enum labels
     * @return enum value
     */
    public static String format(Object data, String labels) {
        if (data == null) {
            return "";
        }
		String[] split = labels.split(",");
		for (int i = 0; i < split.length; i+=2)
			if(split[i].equals(data))
				return split[i+1];
    	return "???<"+data+">";
    }
    
    /**
     * Returns the full filename for a checkbutton that uses image.
     * 
     * @param boolString value of attribute
     * @param selectedImageName filename
     * @param unselectedImageName filename
     * @return full filename
     */
    public static String getButtonImage(String boolString, String selectedImageName, String unselectedImageName) {
        Boolean b= Boolean.valueOf(boolString);
        return (b) ? getImage(selectedImageName) : getImage(unselectedImageName);
    }
    
    /**
     * Gets an image from classpath.
     * 
     * @param filename file
     * @return full filename
     */
    public static String getImage(String filename) {
        URL resourceReference= null;
        try {
            resourceReference= Pathfinder.getResourceReference(
                    IMAGE_PRE_PATH, filename);
            filename= Pathfinder.getAbsoluteFilePath(resourceReference);
        } catch (G9ServiceException e) {
            if (e.getCause() instanceof IllegalArgumentException) {
               // Likely if the image is located in a jar. 
                return (resourceReference==null) ? null : resourceReference.toExternalForm();
            } 
            throw e;
        }
        if (!(new File(filename)).exists()) {
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_COULD_NOT_LOCATE_IMG, filename);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            return null;
        }
        return filename;
    }

    /**
     * Returns filename for imagebox
     *
     * @param b The Boolean to get a graphical representation of
     * @return filename for image that represents b as a checkbox
     */
    public static String getCheckBoxImageFilename(Boolean b) {
        String filename;
        filename= Registry.getRegistry().getG9Property(
                b.booleanValue() ? PROPERTY_FILE_NAME_CHECKBOX_CHECKED
                        : PROPERTY_FILE_NAME_CHECKBOX_UNCHECKED);
        if (filename == null) {
            filename= b.booleanValue() ? CHECKBOX_TRUE_DEFAULT_IMAGE_FILENAME
                    : CHECKBOX_FALSE_DEFAULT_IMAGE_FILENAME;
        }
        URL resourceReference= null;
        try {
            resourceReference= Pathfinder.getResourceReference(
                    IMAGE_PRE_PATH, filename);
            filename= Pathfinder.getAbsoluteFilePath(resourceReference);
        } catch (G9ServiceException e) {
            if (e.getCause() instanceof IllegalArgumentException) {
               // Likely if the image is located in a jar. 
                return (resourceReference==null) ? null : resourceReference.toExternalForm();
            } 
            throw e;
        }
        if (!(new File(filename)).exists()) {
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_COULD_NOT_LOCATE_IMG, filename);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            return null;
        }
        return filename;
    }

    /**
     * Returns filename for imagebox
     *
     * @param s If "true", will get a checked box.
     * @return filename for image that represents b as a checkbox
     */
    public static String getCheckBoxImageFilename(String s) {
        return getCheckBoxImageFilename(Boolean.valueOf(s));
    }

}
