/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.jetty;

import java.io.FileNotFoundException;
import java.net.URL;

import no.g9.support.Pathfinder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.xml.XmlConfiguration;

/**
 * Separate startup class for Jetty.
 * Used when running the generated jVine application in an embedded Jetty instance.
 */
public class JettyStart {

	/** Standard logger. */
    private static Log log;

    static {
		/** Instruct JUL to utilize log4j2 */
		System.setProperty("java.util.logging.manager", "org.apache.logging.log4j.jul.LogManager");
	}

    /**
     * Main function, starts the Jetty server.
     *
     * @param args Command line arguments
     */
    public static void main(String[] args)  {
		log = LogFactory.getLog(JettyStart.class);
        try {
            new JettyStart().go();
        }
        catch (Exception e) {
            log.fatal("Could not start the Jetty server: " + e);
        }
    }

    /**
     * Main run method.
     *
     * @throws Exception on error
     */
    protected void go() throws Exception {
        XmlConfiguration jettyConfig = makeConfig();
        Server jettyServer = new Server();
        setupServer(jettyServer);
        configureServer(jettyConfig, jettyServer);
        try {
            jettyServer.start();
            jettyServer.join();
        }
        catch (Exception e) {
            try {
                jettyServer.stop();
            }
            catch (Exception e1) {
                log.fatal("Unable to stop the jetty server: " + e1);
            }
            throw e;
        }
    }

    /**
     * Setup before configuration is read.
     *
     * @param jettyServer The server
     */
    protected void setupServer(Server jettyServer) {
        jettyServer.setStopAtShutdown(true);
    }

    /**
     * Configure instance.
     *
     * @param jettyConfig Config
     * @param jettyServer Server
     * @throws Exception on error
     */
    protected void configureServer(XmlConfiguration jettyConfig, Server jettyServer) throws Exception {
        jettyConfig.configure(jettyServer);
    }

    /**
     * Create a configuration.
     *
     * @return A finished configuration
     * @throws Exception on error
     */
    protected XmlConfiguration makeConfig() throws Exception {
        URL jettyConfig = Pathfinder.getResourceReference(null, "jetty-config.xml");
        if (jettyConfig == null) {
            throw new FileNotFoundException("Unable to locate jetty-config.xml on the classpath");
        }
        return new XmlConfiguration(jettyConfig);
    }

}
