/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import java.util.HashMap;
import java.util.Map;

import no.esito.log.Logger;
import no.g9.os.OSRole;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.GenericTypeResolver;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

/**
 * A holder for converters used in conversion to and from domain and transport
 * objects.
 * 
 */
@Service
public class DomainTransportConverterRegistry implements ApplicationContextAware, DisposableBean {

    private HashMap<SourceTarget, DomainTransportTransfer<?, ?>> domainTransporConverters;
    private HashMap<OSRole<?>, DomainTransportTransfer<?, ?>> domainTransporConvertersPerOSRole;

    private ApplicationContext applicationContext;
    private final static Logger logger = Logger.getLogger(DomainTransportConverterRegistry.class);

    /**
     * Instantiates a new domain transport converter registry.
     */
    public DomainTransportConverterRegistry() {
        super();
    }

    /**
     * Adds a converter to the registry.
     * 
     * @param <D>
     *            the generic type of the domain class
     * @param <T>
     *            the generic type of the transport class
     * @param converter
     *            the converter to add to registry
     * 
     *            TODO remove
     */
    private synchronized <S, T> void addConverter(DomainTransportTransfer<S, T> converter) {
        Assert.notNull(converter, "converter must not be null");

        Class<?>[] typeInfo = getRequiredTypeInfo(converter);
        SourceTarget convertiblePair1 = new SourceTarget(typeInfo[0], typeInfo[1]);
        SourceTarget convertiblePair2 = new SourceTarget(typeInfo[1], typeInfo[0]);
        getDomainTransporConverters().put(convertiblePair1, converter);
        getDomainTransporConverters().put(convertiblePair2, converter);
        if (logger.isDebugEnabled()) {
            logger.debug("Adding converter for role '" + converter.getOSRole() + "':" + convertiblePair1);
        }
        Assert.notNull(converter.getOSRole(), "converter's osrole must not be null");
        getDomainTransporConvertersPerOSRole().put(converter.getOSRole(), converter);
    }

    /**
     * Determine if a conversion is supported.
     * 
     * @param sourceType
     *            the source type to convert from
     * @param targetType
     *            the target type to convert to.
     * 
     * @return true, if a conversion is supported
     */
    public boolean canConvert(Class<?> sourceType, Class<?> targetType) {
        return getConverter(sourceType, targetType) != null;
    }

    /**
     * Determine if a conversion is supported.
     * 
     * @param sourceType
     *            the source type to convert from
     * @param targetType
     *            the target type to convert to.
     * @return true, if a conversion is supported
     */
    public boolean canConvert(TypeDescriptor sourceType, TypeDescriptor targetType) {
        return getConverter(sourceType, targetType) != null;
    }

    /**
     * Find a converter for a given domain or transport type.
     * 
     * <p> Note that the converters are stored with both transport and domain
     * type as key, so both will give the same result.
     * 
     * @param <S>
     *            The generic type of the source class
     * @param <T>
     *            The generic type of the target class
     * 
     * @param sourceType
     *            The source type
     * @param targetType
     *            The target type
     * 
     * 
     * @return the domain transport converter, <code>null</code> if no converter
     *         is found
     */
    @SuppressWarnings("unchecked")
    public <S, T> DomainTransportTransfer<S, T> getConverter(Class<?> sourceType, Class<?> targetType) {
        SourceTarget typeInfo = new SourceTarget(sourceType, targetType);

        return (DomainTransportTransfer<S, T>) getDomainTransporConverters().get(typeInfo);
    }

    /**
     * @param <S>
     *            The generic type of the source class
     * @param <T>
     *            The generic type of the target class
     * @param sourceType
     *            The type descriptor for the source type
     * @param targetType
     *            The {@link TypeDescriptor} for the target type
     * @see TypeDescriptor
     * @return The converter for the source/target pair.
     */
    @SuppressWarnings("unchecked")
    public <S, T> DomainTransportTransfer<S, T> getConverter(TypeDescriptor sourceType, TypeDescriptor targetType) {
        SourceTarget typeInfo = new SourceTarget(sourceType, targetType);

        return (DomainTransportTransfer<S, T>) getDomainTransporConverters().get(typeInfo);
    }

    /**
     * Get a converter for a given <code>OSRole</code>.
     * 
     * @param <S>
     *            The generic type of the source class
     * @param <T>
     *            The generic type of the target class
     * @param osRole
     *            the <code>OSRole</code> to get the converter for.
     * @return The converter for the source/target pair.
     */
    @SuppressWarnings("unchecked")
    public <S, T> DomainTransportTransfer<S, T> getConverter(OSRole<?> osRole) {
        return (DomainTransportTransfer<S, T>) getDomainTransporConvertersPerOSRole().get(osRole);
    }

    private Class<?>[] getRequiredTypeInfo(Object converter) {
        return GenericTypeResolver.resolveTypeArguments(converter.getClass(), DomainTransportTransfer.class);
    }

    private Map<SourceTarget, DomainTransportTransfer<?, ?>> getDomainTransporConverters() {
        if (domainTransporConverters == null) {
            initConverters();
        }
        return domainTransporConverters;
    }

    private Map<OSRole<?>, DomainTransportTransfer<?, ?>> getDomainTransporConvertersPerOSRole() {
        if (domainTransporConvertersPerOSRole == null) {
            initConverters();
        }
        return domainTransporConvertersPerOSRole;
    }

    /**
     * Load all the converters.
     * 
     * NOTE: Made package scoped for testing.
     */
    @SuppressWarnings("rawtypes")
    synchronized void initConverters() {

        domainTransporConverters = new HashMap<SourceTarget, DomainTransportTransfer<?, ?>>();
        domainTransporConvertersPerOSRole = new HashMap<OSRole<?>, DomainTransportTransfer<?, ?>>();
        if (logger.isDebugEnabled()) {
            logger.debug("initConverters()");
        }
        Map<String, DomainTransportTransfer> beans = getApplicationContext().getBeansOfType(
                DomainTransportTransfer.class);

        for (final DomainTransportTransfer<?, ?> bean : beans.values()) {
            addConverter(bean);
        }

    }

    @Override
    public void destroy() throws Exception {
        domainTransporConverters = null;
    }

    @Override
    public void setApplicationContext(ApplicationContext ctx) throws BeansException {
        this.applicationContext = ctx;
    }

    /**
     * Gets the application context.
     * 
     * @return the application context
     */
    private ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    /**
     * Holder for a source-to-target class pair used for the converter registry.
     */
    public static final class SourceTarget {

        private final Class<?> sourceType;

        private final Class<?> targetType;

        /**
         * Create a new source-to-target pair.
         * 
         * @param sourceType
         *            the source type
         * @param targetType
         *            the target type
         */
        public SourceTarget(Class<?> sourceType, Class<?> targetType) {
            Assert.notNull(sourceType, "Source type must not be null");
            Assert.notNull(targetType, "Target type must not be null");
            this.sourceType = sourceType;
            this.targetType = targetType;
        }

        @SuppressWarnings("javadoc")
        public SourceTarget(TypeDescriptor sourceType, TypeDescriptor targetType) {
            Assert.notNull(sourceType, "Source type must not be null");
            Assert.notNull(targetType, "Target type must not be null");
            this.sourceType = sourceType.getType();
            this.targetType = targetType.getType();

        }

        @SuppressWarnings("javadoc")
        public Class<?> getSourceType() {
            return this.sourceType;
        }

        @SuppressWarnings("javadoc")
        public Class<?> getTargetType() {
            return this.targetType;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + ((sourceType == null) ? 0 : sourceType.hashCode());
            result = prime * result + ((targetType == null) ? 0 : targetType.hashCode());
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            SourceTarget other = (SourceTarget) obj;
            if (sourceType == null) {
                if (other.sourceType != null)
                    return false;
            } else if (!sourceType.equals(other.sourceType))
                return false;
            if (targetType == null) {
                if (other.targetType != null)
                    return false;
            } else if (!targetType.equals(other.targetType))
                return false;
            return true;
        }

        @Override
        public String toString() {
            return "SourceTarget [sourceType=" + sourceType + ", targetType=" + targetType + "]";
        }

    }

}
