/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import java.io.IOException;

import javax.xml.transform.Result;
import javax.xml.transform.Source;

import no.g9.os.OSRole;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.convert.ConversionService;
import org.springframework.oxm.Marshaller;
import org.springframework.oxm.Unmarshaller;
import org.springframework.oxm.XmlMappingException;
import org.springframework.util.Assert;

/**
 * A <code>Marshaller</code> and <code>Unmarshaller</code> that handles the
 * conversion to the <code>targetType</code>. The actual marshalling and
 * unmarshalling is delegated to the supplied <code>Marshaller</code> and
 * <code>Unmarshaller</code>.
 * 
 */
public class DomainTransportMarshaller implements Marshaller, Unmarshaller, InitializingBean {

    private Marshaller marshaller;
    private Unmarshaller unmarshaller;
    private ConversionService domainTransportConversionService;
    private Class<?> transportType;
    private Class<?> domainType;
    private OSRole<?> role;

    /**
     * Steps performed:
     * <ol>
     * <li>Delegates unmarshal to the supplied <code>unmarshaller</code></li>
     * <li>Converts the unmarshalled object graph to the configured
     * <code>domainType</code></li>
     * </ol>
     * 
     * @see org.springframework.oxm.Unmarshaller#unmarshal(javax.xml.transform.Source)
     */
    @Override
    public Object unmarshal(final Source source) throws IOException, XmlMappingException {
        Object unmarshalled = getUnmarshaller().unmarshal(source);
        return getDomainTransportConversionService().convert(unmarshalled, getDomainType());
    }

    /**
     * Steps performed:
     * <ol>
     * <li>Converts the object graph to the configured
     * <code>transportType</code></li>
     * <li>Delegates marshal to the supplied <code>unmarshaller</code></li>
     * </ol>
     * 
     * @see org.springframework.oxm.Marshaller#marshal(java.lang.Object,
     *      javax.xml.transform.Result)
     */
    @Override
    public void marshal(final Object graph, final Result result) throws IOException, XmlMappingException {
        Object convertedGraph = getDomainTransportConversionService().convert(graph, getTransportType());
        getMarshaller().marshal(convertedGraph, result);
    }

    @Override
    public boolean supports(Class<?> clazz) {
        return true;
    }

    private Marshaller getMarshaller() {
        return marshaller;
    }

    /**
     * Sets the marshaller to use.
     * 
     * @param marshaller
     *            the marshaller to use
     */
    @Required
    public void setMarshaller(Marshaller marshaller) {
        this.marshaller = marshaller;
    }

    private Unmarshaller getUnmarshaller() {
        return unmarshaller;
    }

    /**
     * Sets the unmarshaller to use.
     * 
     * @param unmarshaller
     *            the unmarshaller to use
     */
    @Required
    public void setUnmarshaller(Unmarshaller unmarshaller) {
        this.unmarshaller = unmarshaller;
    }

    private ConversionService getDomainTransportConversionService() {
        return domainTransportConversionService;
    }

    /**
     * Sets the conversion service used to convert to the target type.
     * 
     * @param domainTransportConversionService
     *            the conversion service to use
     */
    @Required
    public void setDomainTransportConversionService(ConversionService domainTransportConversionService) {
        this.domainTransportConversionService = domainTransportConversionService;
    }

    private Class<?> getTransportType() {
        return transportType;
    }

   
    private Class<?> getDomainType() {
        return domainType;
    }

 
    /**
     * @return the <code>OSRole</code> used for conversion to/from domain objects.
     */
    public OSRole<?> getRole() {
        return role;
    }

    /**
     * @param role the <code>OSRole</code> to use for conversion to/from domain objects.
     */
    @Required
    public void setRole(OSRole<?> role) {
        this.role = role;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.isInstanceOf(DomainTransportConversionService.class, getDomainTransportConversionService());
        DomainTransportTransfer<?, ?> transfer = ((DomainTransportConversionService) getDomainTransportConversionService())
                .getConverterForRole(this.role);
        Assert.notNull(transfer, "No DomainTransportConversionService found");
        this.domainType = transfer.getDomainType();
        this.transportType = transfer.getTransportType();
    }

}
