/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.support.G9Enumerator;

import org.springframework.beans.BeanUtils;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.converter.ConverterFactory;

/**
 * A factory for conversion of G9Enum objects to Java enums.
 *
 * <strong>WARNING:</strong> Although this class is public, it should not be treated as part of the public API, as it
 * might change in incompatible ways between releases (even patches).
 * <p>
 * <strong>NOTE:</strong> This factory depends on a static method called <code>ordinalToEnum</code> on the target type.
 */
@SuppressWarnings("rawtypes")
public class G9EnumToJavaEnumConverterFactory implements ConverterFactory<G9Enumerator, Enum> {

    /** The static factory method used to create a new target Enum. */
    private static final String FACTORY_METHOD_NAME = "get";

    /*
     * (non-Javadoc)
     *
     * @see org.springframework.core.convert.converter.ConverterFactory#getConverter (java.lang.Class)
     */
    @Override
    public <T extends Enum> Converter<G9Enumerator, T> getConverter(Class<T> targetType) {
        return new G9EnumToJavaEnumConverter<T>(targetType);
    }

    /**
     * The Class G9EnumToJavaEnumConverter.
     *
     * @param <T>
     *            the generic type
     */
    private final static class G9EnumToJavaEnumConverter<T extends Enum> implements Converter<G9Enumerator, T> {

        /** The enum type. */
        private Class<T> enumType;

        /** The static factory method for enumType. */
        Method factoryMethod = null;

        /**
         * Instantiates a new g9 enum to java enum converter.
         *
         * @param enumType
         *            the enum type
         */
        public G9EnumToJavaEnumConverter(Class<T> enumType) {
            this.enumType = enumType;
            this.factoryMethod = BeanUtils.findMethod(this.enumType, FACTORY_METHOD_NAME, int.class);
            if (this.factoryMethod == null) {
                handleException(FACTORY_METHOD_NAME, new Exception("G9EnumToJavaEnumConverter constructor failed"));
            }
        }

        /*
         * (non-Javadoc)
         *
         * @see org.springframework.core.convert.converter.Converter#convert(java .lang.Object)
         */
        @Override
        @SuppressWarnings("unchecked")
        public T convert(G9Enumerator source) {
            Exception ex = null;
            try {
                return (T) factoryMethod.invoke(null, source.currentValue);
            } catch (IllegalArgumentException e) {
                ex = e;
            } catch (IllegalAccessException e) {
                ex = e;
            } catch (InvocationTargetException e) {
                ex = e;
            }
            handleException(FACTORY_METHOD_NAME, ex);

            return null;

        }

        private void handleException(String methodOrField, Exception ex) {
            Object[] msgArgs = { enumType.getClass(), methodOrField, this.getClass(), ex.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD,
                            msgArgs);
            throw new G9ServiceException(ex, msg);
        }
    }

}
