/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.spring.aspect;

import no.esito.log.Logger;
import no.g9.exception.G9BaseException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

import org.springframework.aop.ThrowsAdvice;
import org.springframework.remoting.RemoteAccessException;

/**
 * Intercepts all exceptions thrown by services.
 * <p><strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 */
public class ExceptionHandlerAspect implements ThrowsAdvice {

    /** Logger. */
    private static Logger log = Logger.getLogger(ExceptionHandlerAspect.class);

    /**
     * Default empty constructor.
     */
    public ExceptionHandlerAspect() {
        super();
        log.debug("Finished constructing ExceptionHandlerAspect.");
    }

    /**
     * Exception transformer. The following steps are taken 1) If
     * RemoteAccessException, the wrapping exception is discarded 2) A
     * G9BaseException is thrown, with its cause being the input throwable
     * 
     * @param e The Throwable to transform into a G9BaseException
     * @throws G9BaseException In all cases of execution
     */
    public void afterThrowing(Throwable e) throws G9BaseException {
        Throwable ex = e;
        if (ex instanceof RemoteAccessException) {
            ex = unwrap(ex);
        }
        if (ex instanceof G9BaseException)
            throw (G9BaseException) ex;
        throw new G9BaseException(ex, makeMessage(ex));
    }

    /**
     * Returns the cause of the throwable, unless the cause is null, in which
     * case the input Throwable is returned.
     * 
     * @param e the Throwable to unwrap
     * @return the cause of the Throwable, or the Throwable itself if no cause
     */
    private Throwable unwrap(Throwable e) {
        return e.getCause() == null ? e : e.getCause();
    }

    /**
     * Creates the appropriate message for the exception.
     * 
     * @param exception the exception to create the message for
     * @return the message
     */
    private Message makeMessage(Throwable exception) {
        return MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.UNHANDLED_EXCEPTION, exception.getMessage());
    }
}
