/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.spring.aspect;

import no.esito.log.Logger;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;

/**
 * Intercepts all service calls and logs elapsed time.
 * 
 */
public class MethodCallProfilerAspect implements MethodInterceptor{

    /** Logger */
    private static Logger log = Logger.getLogger(MethodCallProfilerAspect.class);

    /**
     * Default empty constructor.
     */
    public MethodCallProfilerAspect() {
        super();
        log.debug("Finished constructing MethodCallProfilerAspect.");
    }

    /**
     * Invokes service call, and does some additional logging.
     * 
     * @param invocation The service call join point
     * @return The result of the service call
     * @throws Throwable exceptions raised during invocation is re-thrown.
     */
    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {
        long time = 0;
        String invokeString = null;
        boolean throwableThrown = false;
        if (log.isDebugEnabled()) {
            invokeString = invocation.getThis().getClass().getCanonicalName()
                + "." + invocation.getMethod().getName();
            log.debug("Invoking " + invokeString);
            time = System.currentTimeMillis();
        }
        try {
            return invocation.proceed();
        } catch (Throwable e) {
            throwableThrown = true;
            throw e;
        } finally {
            if (log.isDebugEnabled()) {
                time = System.currentTimeMillis() - time;
                log.debug(invokeString + 
                        (throwableThrown ? " failed" : " returned") 
                        + " in " + time + " milliseconds.");
            }
        }
    }

}
