/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.util;

import no.esito.log.Logger;
import no.g9.exception.G9BaseException;
import no.g9.service.G9Spring;
import no.g9.support.ObjectFactory;
import no.g9.support.Registry;

/**
 * Utility class for loading services.
 * 
 * <p>
 * <strong>WARNING:</strong> Although this class is public, it should not be
 * treated as part of the public API, as it might change in incompatible ways
 * between releases (even patches).
 * 
 */
public final class ServiceLoader {

	/**
	 * Private constructor - only static methods available.
	 */
	private ServiceLoader() {
		// Only static methods.
	}

	/** Logger. */
	private static Logger log = Logger.getLogger(ServiceLoader.class);

	/**
	 * Returns the bean id annotation value of the specified class.
	 * 
	 * @param <T>
	 *            The service interface type
	 * @param clazz
	 *            the class specifying a service
	 * @return the bean id
	 */
	private static <T> String beanID(Class<T> clazz) {
		BeanID beanID = clazz.getAnnotation(BeanID.class);
		return beanID != null ? beanID.value() : null;
	}

	/**
	 * Returns the g9 property name annotation value of the specified class.
	 * 
	 * @param <T>
	 *            The service interface type
	 * @param clazz
	 *            the class specifying a service
	 * @return the property name
	 */
	private static <T> String propertyName(Class<T> clazz) {
		G9PropertyName propName = clazz
				.getAnnotation(G9PropertyName.class);
		return propName != null ? propName.value() : null;
	}

	/**
	 * Loads a service, either instantiates the service class defined in the the
	 * property settings, or loads the default service bean.
	 * 
	 * @param <T>
	 *            the type of the service interface
	 * @param serviceInterface
	 *            the service interface class
	 * @return the service
	 */
	public static <T> T getService(Class<T> serviceInterface) {
		T service = null;
		log.info("Instantiating " + serviceInterface.getSimpleName());
		String propertyName = propertyName(serviceInterface);
		if (propertyName != null) {
			Registry registry = null;
			try {
				registry = Registry.getRegistry();
			} catch (G9BaseException e) {
				log.warn(e.getErrMsg());
			}
			if (registry != null && registry.hasG9Propery(propertyName)) {
				if (log.isDebugEnabled()) {
					log.debug("Found property for " + propertyName);
				}
				String className = Registry.getRegistry().getG9Property(
						propertyName);
				if (log.isDebugEnabled()) {
					log.debug("Setting up " + propertyName + " using "
							+ className);
				}
				Object obj = ObjectFactory.newObject(className);
				service = serviceInterface.cast(obj);
				return service;
			}
		}

		//Try to load a bean.		
		String beanID = beanID(serviceInterface);
		if (beanID == null) {
			log.error("Missing both property and bean id for "
					+ serviceInterface.getCanonicalName());
			throw new G9BaseException("Failed to load service "
					+ serviceInterface.getSimpleName());
		}
		if (log.isDebugEnabled()) {
			log.debug("Setting up " + propertyName + " as spring bean: \""
					+ beanID + "\"");
		}
		try {
			service = G9Spring.getBean(serviceInterface, beanID);
		} catch (Exception e) {
			log.error("Failed to load bean. Hint: Check class path.", e);
			throw new G9BaseException(e);
		}

		return service;

	}

}
