/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.domain;

import java.util.HashMap;
import java.util.Map;


/**
 * Provider for DomainObjectProxy objects.
 */
@SuppressWarnings({ "unchecked", "rawtypes" })
public class DomainProxy {

    private  Map<ProxyKey, DomainObjectProxy> proxiedObjects = new HashMap<ProxyKey, DomainObjectProxy>();
    private boolean useAccessMethods;
    private G9Enums enums;

    /**
     * Constructs a new instance of <code>DomainProxy</code>, which can be used
     * to create new DomainObjectProxies.
     * @param useAccessMethods flag singalling if attributes should be accessed
     * through bean-style get- and set-methods. If <code>true</code> get- and
     * set-methods are used.
     *
     * @param enums a reference to the application enums.
     */
    public DomainProxy(boolean useAccessMethods, G9Enums enums) {
        this.useAccessMethods = useAccessMethods;
        this.enums = enums;
    }


    /**
     * Returns the proxy associeated with the specified object. If no such proxy exist,
     * a new proxy is constructed and returned (which then defaults to clean state).
     * @param <T> Type of proxied domain object
     * @param o the object to proxy
     * @param roleName the role name of the object
     * @return the associated proxy for the specified object.
     */
    public <T> DomainObjectProxy<T> getProxy(T o, String roleName) {
        ProxyKey key = new ProxyKey(o, roleName);
        DomainObjectProxy<T> d = proxiedObjects.get(key);
        if (d == null) {
            d = new G9Proxy<T>(o, enums, useAccessMethods);
            proxiedObjects.put(key, d);
        }
        return d;
    }


    /**
     * Returns a new proxy for the domain object
     * @param <T> Type of proxied domain object
     * @param o the domain object
     * @param roleName the role name of the domain object
     * @return a new proxy
     */
    public <T> DomainObjectProxy<T> getNewProxy(T o, String roleName) {
        G9Proxy<T> g = new G9Proxy<T>(o, enums, useAccessMethods);
        proxiedObjects.put(new ProxyKey(o, roleName), g);
        return g;
    }

    /**
     * Removes a proxy from the list of registred proxies.
     * @param proxy the proxy to remove
     */
    public  void removeProxy(DomainObjectProxy<?> proxy) {
        proxiedObjects.values().remove(proxy);
    }

    /**
     * Removes all proxies regiestred in this proxy holder.
     *
     */
    public void removeAllProxies() {
        proxiedObjects.clear();
    }

    private static class ProxyKey {
        private Object domainObject;
        private String roleName;

        /**
         * Constructs a new proxy key object
         * @param domainObject the domain object
         * @param roleName the role name of the domain object
         */
        public ProxyKey(Object domainObject, String roleName) {
            this.domainObject = domainObject;
            this.roleName = roleName;
        }

        @Override
        public int hashCode() {
            return domainObject.hashCode() + roleName.hashCode();
        }

        @Override
        public boolean equals(Object o) {
            if (o == null || (!(o instanceof ProxyKey))) {
                return false;
            }

            ProxyKey tmp = (ProxyKey) o;
            return roleName.equals(tmp.roleName) && domainObject.equals(tmp.domainObject);

        }

    }

}






