/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.exception;

import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 * The general execption class used in g9.
 */
public class G9BaseException extends RuntimeException {

    private Message errMsg;

	/**
	 * @param cause (missing javadoc)
	 */
	public G9BaseException(Throwable cause) {
        super(cause);
        this.setErrMsg(MessageSystem.getMessageFactory().getMessage(
                CRuntimeMsg.UNHANDLED_EXCEPTION, cause.getLocalizedMessage()));
        this.getErrMsg().setException(cause);
	}

	/**
	 * @param message (missing javadoc)
	 * @param cause (missing javadoc)
	 */
	public G9BaseException(String message, Throwable cause) {
		super(message, cause);
		this.setErrMsg(MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.UNHANDLED_EXCEPTION, message));
		this.getErrMsg().setException(cause);
	}

	/**
	 * @param message (missing javadoc)
	 */
	public G9BaseException(String message) {
		super(message);
		this.setErrMsg(MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.UNHANDLED_EXCEPTION, message));
	}

	/**
	 *  (missing javadoc)
	 */
	public G9BaseException() {
		super();
	}

    /**
     * Constructor with an existing Message.
     *
     * @param errMsg Message
     */
	public G9BaseException(final Message errMsg) {
		super(errMsg != null ? errMsg.getMessageText() : null);
		this.setErrMsg(errMsg);
	}

    /**
     * Constructor with an existing Message and a wrapped exception.
     *
     * @param cause Nested exception
     * @param errMsg Message
     */
	public G9BaseException(final Throwable cause, final Message errMsg) {
		super(errMsg != null ? errMsg.getMessageText() : null, cause);
		this.setErrMsg(errMsg);
		this.getErrMsg().setException(cause);
	}

    /**
     * Store a Message.
     *
     * @param errMsg New value of errMsg.
     * @see #getErrMsg()
     */
	public final void setErrMsg(final Message errMsg) {
		this.errMsg = errMsg;
	}

    /**
     * Returns the current error message. NOTE: original object, not copied.
     *
     * @return The stored Message object, if any.
     * @see #setErrMsg(Message)
     */
	public final Message getErrMsg() {
		return this.errMsg;
	}

    /**
     * Check if a given exception is of the same type as this class (including
     * subclasses)
     *
     * @param cause The exception to check
     * @return <code>null</code> if no nested exception is compatible, with this
     *         class, otherwise the outermost such exception in the chain
     */
	public static Throwable hasNested(final Throwable cause) {
		Throwable t = cause;
		while (true) {
			if (t == null) {
				return null;
			}
			if (t instanceof G9BaseException) {
				return t;
			}
			t = t.getCause();
		}
	}

}
