/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.exception;

/**
 * A general exception, suitable for extending your own. This exception is
 * understood by the generated code from g9.
 */
public class GeneralException extends Exception {

    /** The error code */
    protected int errorCode;

    /** The error message */
    protected String errorMessage;

    /** The sub system */
    protected int subSystem;

    /** log exception flag */
    protected boolean logException;

    /** logged exception flag */
    protected boolean exceptionIsLogged;

    /** the serious exception code */
    protected int exceptionSeriousCode;

    /** the original exception */
    protected Throwable originalException;

    /** Constant denoting a warning */
    public final static int WARNING = 1;

    /** Constant denoting an error */
    public final static int ERROR = 2;

    /** Constant denoting a fatal error */
    public final static int FATAL_ERROR = 3;

    /**
     * Returns error code.
     * 
     * @return the error code
     */
    public int getErrorCode() {
        return errorCode;
    }

    /** Default empty constructor */
    public GeneralException() {
        // Empty.
    }

    @Override
    public String getMessage() {
        return errorMessage;
    }

    /**
     * Returns error message.
     * 
     * @return the error message
     */
    public String getErrorMessage() {
        return errorMessage;
    }

    /**
     * Returns subsystem where this error occured.
     * 
     * @return the sub system
     */
    public int getSubSystem() {
        return subSystem;
    }

    /**
     * Returns true if the exception should be logged
     * 
     * @return <code>true</code> if exception should be logged.
     */
    public boolean shouldExceptionBeLogged() {
        return logException;
    }

    /**
     * Returns true if the exception has been logged
     * 
     * @return <code>true</code> if exception is logged.
     */
    public boolean exceptionLogged() {
        return exceptionIsLogged;
    }

    /**
     * Returns the seriousnes of exception. Should be one of WARNING, ERROR or
     * FATAL_ERROR
     * 
     * @return the seriousnes of the exception
     */
    public int exceptionSeriousness() {
        return exceptionSeriousCode;
    }

    /**
     * Returns the seriousnes of exception as a string
     * 
     * @return String representation of the seriousness of the exception
     */
    public String exceptionSeriousnessToString() {
        switch (exceptionSeriousCode) {
            case WARNING:
                return "warning";
            case ERROR:
                return "error";
            case FATAL_ERROR:
                return "fatal error";
        }
        return "uncategorized error";
    }

    /**
     * Returns the original error which triggered this exception, if it
     * exist
     * 
     * @return the original exception
     */
    public Throwable getOriginalException() {
        return originalException;
    }

    /**
     * Construcuts a new GeneralException. When the constructor is run, the
     * method log() is called.
     * 
     * @param errorCode The error code for the exception.
     * @param errorMessage The error message for the exception. This message is
     *            to be displayed to the user, so it should be user-friendly.
     * @param subsystem The subsystem of where the exception originated.
     * @param shouldBeLogged Flags if the exception should be logged.
     * @param seriousness The seriousness of the exception. Should be one of
     *            WARNING, ERROR or FATAL_ERROR.
     * @param originalException The exception leading to this exception, if any.
     */
    public GeneralException(int errorCode, String errorMessage,
            int subsystem, boolean shouldBeLogged, int seriousness,
            Throwable originalException) {
        this.errorCode = errorCode;
        this.errorMessage = errorMessage;
        this.subSystem = subsystem;
        this.logException = shouldBeLogged;
        this.exceptionSeriousCode = seriousness;
        this.originalException = originalException;
        log();
    }

    /** Override this to make your own logging mechanism */
    protected void log() {
        if (!logException || exceptionIsLogged) {
            return;
        }
        System.out.println(toString());
        exceptionIsLogged = true;
    }

    
    @Override
    public String toString() {
        return exceptionSeriousnessToString() + " - CODE:" + errorCode + ":" + getMessage();
    }

    /**
     * Invokes toString method of the Exception class.
     * 
     * @return a string representation of this exception.
     * @see Exception#toString()
     */
    public String toStringSuper() {
        return super.toString();
    }
}
