/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import org.springframework.beans.factory.annotation.Autowired;

/**
 * Standard implementation of the <code>MessageDispatcher</code>
 */
public class MessageDispatcherImpl implements MessageDispatcherSetup {

    private MessageLogDispatcher messageLogDispatcher;
    private MessageInteractor messageInteractor;
    private DispatcherContext dispatcherContext;
    private DispatcherContextProvider dispatcherContextProvider;

    /**
     * Set the message log dispatcher used by this dispatcher.
     *
     * @param messageLogDispatcher the message log dispatcher that will be used to dispatch messages to the log
     */
    @Autowired(required = false)
    public void setMessageLogDispatcher(MessageLogDispatcher messageLogDispatcher) {
        this.messageLogDispatcher = messageLogDispatcher;
    }

    /**
     * Set the dispatcher context provider, which will be used to get a dispatcher context
     * if this dispatcher doesn't have one.
     *
     * @param dispatcherContextProvider the dispatcher context provider to use
     */
    @Autowired(required = false)
    public void setDispatcherContextProvider(DispatcherContextProvider dispatcherContextProvider) {
    	this.dispatcherContextProvider = dispatcherContextProvider;
    }

    /**
     * Set the message interactor used by this dispatcher.
     *
     * @param messageInteractor the interactor that will (possibly) display messages to the end user.
     */
    public void setMessageInteractor(MessageInteractor messageInteractor) {
        this.messageInteractor = messageInteractor;
    }


    @Override
    public void setDispatcherContext(DispatcherContext dispatcherContext) {
        this.dispatcherContext = dispatcherContext;
    }

    @Override
    public MessageReply dispatch(String id, Object... args) {
        Message message = MessageSystem.getMessageFactory().getMessage(id, args);
        return dispatch(message);
    }

    @Override
    public MessageReply dispatch(Message message) {
        DispatcherContext currentDispatcherContext = getDispatcherContext();
		if(messageLogDispatcher != null) {
            messageLogDispatcher.log(message, currentDispatcherContext);
        }
        MessageReply reply = null;
        if (messageInteractor != null) {
            if(MessageSystem.NO_INTERACTION != currentDispatcherContext){
                reply = messageInteractor.interact(message, currentDispatcherContext);
            }
        }
        return reply;
    }

    @Override
    public MessageReply dispatch(String id, Throwable exception, Object... args) {
        Message message = MessageSystem.getMessageFactory().getMessage(id, args);
        if (exception != null) {
            message.setException(exception);
        }
        return dispatch(message);
    }

    private DispatcherContext getDispatcherContext() {
    	if (dispatcherContext == null) {
    		if (dispatcherContextProvider != null) {
    			return dispatcherContextProvider.getDispatcherContext();
    		}
    		return MessageSystem.NO_INTERACTION; // The default context if there is no context provider
    	}
    	return dispatcherContext;
    }
    
}