/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.message;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Representation of a message text in its various forms...
 */
public class MessageText implements Serializable, Cloneable {

    /** Initial capacity for {@link #parsed}. */
    private static final int INITCAP = 4;

    /** Cache identifier. */
    private final Object id;

    /**
     * The original, unparsed text. May be cleared after parsing, to conserve
     * space.
     */
    private String raw;

    /** The raw text parsed into fixed and variable sections. Never null. */
    private final List<ITextPart> parsed = new ArrayList<ITextPart> (INITCAP);

    /**
     * The result of substitution. Never NULL, may be empty or equal to
     * <code>raw</code>.
     */
    private String expanded = "";

    /**
     * Default constructor.
     */
    public MessageText() {
        this(null, "");
    }

    /**
     * Constructor for MessageText identified by number.
     * 
     * @param messageID message ID for this text
     * @param pRawText Unparsed text
     */
    public MessageText(String messageID, final String pRawText) {
        super();
        String rawText = (pRawText == null) ? "" : pRawText;
        id = messageID;
        this.raw = rawText;
        this.expanded = rawText;
    }

    /**
     * Constructor for uncached + unparsed MessageTexts.
     * 
     * @param text The text...
     */
    public MessageText(final String text) {
        this(null, text);
        this.parsed.add(new FixedTextPart(text));
    }

    /**
     * Makes a copy of this object. Overrides Object.clone.
     * 
     * @return A shallow copy of this object
     * @throws CloneNotSupportedException Never
     * @see java.lang.Object#clone()
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    /**
     * Returns hash code based on ID.
     * 
     * @return Hash code
     * @see java.lang.Object#clone()
     */
    @Override
    public int hashCode() {
        return this.getId().hashCode();
    }

    /**
     * Checks equality based on ID. Overrides Object.equals.
     * 
     * @param that Object to compare to
     * @return TRUE if objects are Equal
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(final Object that) {
        if (that == null) return false;
        if (that == this) return true;
        if (!(that.getClass().equals(this.getClass()))) return false;
        return this.getId().equals(((MessageText) that).getId());
    }

    /**
     * Stores unparsed text. Also stored as expanded text, until
     * <code>substitute()</code> is run.
     * 
     * @param pRaw New text.
     * @see #getRaw()
     */
    void setRaw(final String pRaw) {
        this.raw = pRaw;
        this.expanded = pRaw;
    }

    /**
     * Returns the raw (unparsed) text.
     * 
     * @return Unparsed text string.
     * @see #setRaw(String)
     */
    String getRaw() {
        return this.raw;
    }

    /**
     * Adds to the list of parsed elements.
     * 
     * @param part New parsed element.
     * @see #getParsed()
     */
    void appendParsed(final ITextPart part) {
        if (part != null) {
            this.parsed.add(part);
        }
    }

    /**
     * Returns an unmodifiable list of parsed elements. Note that the elements
     * are not copies, handle with care!
     * 
     * @return The parsed list.
     * @see #appendParsed(ITextPart)
     */
    List<ITextPart> getParsed() {
        return Collections.unmodifiableList(this.parsed);
    }

    /**
     * Returns the ID of this object.
     * 
     * @return The ID.
     */
    public Object getId() {
        return this.id;
    }

    /**
     * Parses and keeps a substituted string for this text (which is also
     * returned).
     * 
     * @param singleLine As specified in
     *            {@link ITextPart#appendPart(StringBuffer, boolean, Object[])}
     * @param args Substitution variables
     * @return Message text.
     */
    public String substitute(final boolean singleLine, final Object[] args) {
        this.expanded = "";
        if (this.parsed == null) {
            if (this.raw == null) {
                return "";
            }
            this.expanded = this.raw;
            return this.expanded;
        }
        if (this.parsed.isEmpty()) {
            StdParser.parse(this.raw, this.parsed);
        }
        final StringBuffer sb = new StringBuffer(80);
        for (ITextPart elem : parsed) {
            elem.appendPart(sb, singleLine, args);
        }
        this.expanded = sb.toString();
        return this.expanded;
    }

    /**
     * Returns the "best possible representation" of this object as a String - a
     * substituted string if substitution has been performed, otherwise the raw
     * text (or empty).
     * 
     * @return This object as a String.
     */
    @Override
    public String toString() {
        return this.expanded;
    }
}
