/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.os;


/**
 * Representation of a key.
 */
public final class Key {

    /**
     * The key name.
     */
    private final String keyName;

    /**
     * The role.
     */
    private final RoleConstant roleConstant;

    /**
     * The key attributes.
     */
    private final AttributeConstant[] attributeConstants;

    /** Stored hash code */
    private volatile int hashCode;

    /**
     * Construct a new Key with the specified name and attributes.
     * 
     * @param roleConstant the role constant.
     * @param keyName the name of the key.
     * @param attributes an array of attributes.
     */
    public Key(RoleConstant roleConstant, String keyName,
            AttributeConstant... attributes) {
        this.roleConstant = roleConstant;
        this.keyName = keyName;
        this.attributeConstants = attributes;
    }

    /**
     * The name of the key.
     * 
     * @return the key name.
     */
    public final String getKeyName() {
        return keyName;
    }

    /**
     * Get the role this key belongs to.
     * 
     * @return the role constant.
     */
    public RoleConstant getRoleConstant() {
        return roleConstant;
    }

    /**
     * Get the attributes of this key.
     * 
     * @return the key's attributes.
     */
    public final AttributeConstant[] getAttributes() {
        return attributeConstants.clone();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Key)) {
            return false;
        }

        Key that = (Key) obj;

        if (!roleConstant.equals(that.roleConstant)) {
            return false;
        }

        if (!keyName.equals(that.keyName)) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {

        int result = hashCode;
        if (result == 0) {
            result = 19;
            result = 31 * result + roleConstant.hashCode();
            result = 31 * result + keyName.hashCode();
            hashCode = result;
        }

        return result;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(roleConstant).append(".").append(keyName).append(", [");
        for (int i = 0; i < attributeConstants.length; i++) {
            sb.append(attributeConstants[i]);
            if (i < attributeConstants.length - 1) {
                sb.append(", ");
            }
        }
        sb.append("]");
        return sb.toString();
    }

}
