/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service.print;

import java.io.File;
import java.io.OutputStream;
import java.net.URI;
import java.util.Map;

import no.esito.util.BeanID;
import no.esito.util.G9PropertyName;

import org.w3c.dom.Document;

/**
 * The g9 print interface, defining export and print methods.
 */
@BeanID("g9Print")
@G9PropertyName("G9Print")
public interface G9Print {

    /**
     * Carries out the actual filling and printing.
     *
     * @param data          Data formed in a document
     * @param params        Parameters
     * @param dialogName    The dialog to be printed
     * @param printerName   The name of the printer to print to
     * @return              <code>true</code> if OK is pressed in the print
     *                      dialog or printing to an explicit printer succeeded.
     */
    public abstract boolean printReport(Document data,
            Map<String, Object> params, String dialogName, String printerName);

    /**
     * Carries out the actual filling and printing.
     *
     * @param data          Data referenced from an URI
     * @param params        Parameters
     * @param dialogName    The dialog to be printed
     * @param printerName   The name of the printer to print to
     * @return              <code>true</code> if OK is pressed in the print
     *                      dialog or printing to an explicit printer succeeded.
     */
    public abstract boolean printReport(URI data, Map<String, Object> params,
            String dialogName, String printerName);

    /**
     * Carries out the actual filling and printing.
     *
     * @param data          Data formed in an XML
     * @param params        Parameters
     * @param dialogName    The dialog to be printed
     * @param printerName   The name of the printer to print to
     * @return              <code>true</code> if OK is pressed in the print
     *                      dialog or printing to an explicit printer succeeded.
     */
    public abstract boolean printReport(String data,
            Map<String, Object> params, String dialogName, String printerName);

    /**
     * Carries out the filling and the export of the pdf.
     *
     * @param data          Document containing data
     * @param dialogName    Name of the dialog
     * @param file          File name to place the pdf in
     * @param os            OutputStream to place the pdf in
     * @param params        Parameters to the filling of the report
     */
    public abstract void exportPDF(Document data, String dialogName, File file,
            OutputStream os, Map<String, Object> params);

}